#![allow(dead_code)]
use super::{
    bincode::{deserialize, serialize},
    serde::{Deserialize, Serialize},
    toml::{de, from_str, ser, to_string_pretty},
};

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct Configuration {
    pub template_dir: String,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct Template {
    pub language: String,

    pub projects_dir: String,
    pub editor_open: String,

    pub commands: Vec<String>,
    pub copy_files: Vec<(String, String)>,
}

impl Configuration {
    pub fn new(template_dir: String) -> Self {
        Self { template_dir }
    }

    pub fn from_binary(source: &[u8]) -> Option<Self> {
        let config = deserialize(source);

        if config.is_err() {
            eprintln!("Error: {:?}", config.expect_err("").as_ref().clone());
            return None;
        }

        Some(config.expect(""))
    }

    pub fn to_binary(&self) -> Option<Vec<u8>> {
        let binary = serialize(self);

        if binary.is_err() {
            eprintln!("Error: {:?}", binary.expect_err("").as_ref().clone());
            return None;
        }

        Some(binary.expect(""))
    }
}

impl Template {
    pub fn new(
        language: String,
        projects_dir: String,
        editor_open: String,
        commands: Vec<String>,
        copy_files: Vec<(String, String)>,
    ) -> Self {
        Self {
            language,
            editor_open,
            projects_dir,
            commands,
            copy_files,
        }
    }

    //Binary Parsing

    pub fn from_binary(source: &[u8]) -> Option<Self> {
        let config = deserialize(source);

        if config.is_err() {
            eprintln!("Error: {:?}", config.expect_err("").as_ref().clone());
            return None;
        }

        Some(config.expect(""))
    }

    pub fn to_binary(&self) -> Option<Vec<u8>> {
        let binary = serialize(self);

        if binary.is_err() {
            eprintln!(
                "Error: {:?}",
                binary.expect_err("Failed to get error").as_ref().clone()
            );
            return None;
        }

        Some(binary.expect(""))
    }

    // TOML Parsing

    pub fn from_toml(source: String) -> Option<Self> {
        let template: Result<Self, de::Error> = from_str(&source);

        if let Ok(template) = template {
            return Some(template);
        }

        eprintln!("Error {:?}", template.expect_err("Failed to get error"));

        return None;
    }

    pub fn to_toml(&self) -> Option<String> {
        let string: Result<String, ser::Error> = to_string_pretty(self);

        if let Ok(string) = string {
            return Some(string);
        }

        eprintln!("Error {:?}", string.expect_err("Failed to get error"));

        return None;
    }
}
