/*
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

use crate::edit::{EditResponse, SaveOptions, Saveable};
use crate::parsoid::ImmutableWikicode;
use crate::{Bot, Error, Result, Title};
use mwapi_responses::prelude::*;
use once_cell::sync::OnceCell;
use serde_json::Value;
use std::sync::{Arc, Mutex as SyncMutex};
use tokio::sync::OnceCell as AsyncOnceCell;
use tracing::info;

/// Represents a wiki page and provides accessors and mutators.
#[derive(Debug, Clone)]
pub struct Page {
    pub(crate) bot: Bot,
    pub(crate) title: Title,
    pub(crate) title_text: OnceCell<String>,
    pub(crate) info: Arc<AsyncOnceCell<InfoResponseItem>>,
    pub(crate) baserevid: Arc<SyncMutex<Option<u64>>>,
}

#[query(prop = "info", inprop = "associatedpage|url")]
pub(crate) struct InfoResponse {}

impl Page {
    /// Get the title of the page
    pub fn title(&self) -> &str {
        self.title_text.get_or_init(|| {
            let codec = &self.bot.config.codec;
            codec.to_pretty(&self.title)
        })
    }

    /// Get the namespace ID of the page
    pub fn namespace(&self) -> i32 {
        self.title.namespace()
    }

    /// Whether this page refers to a file
    pub fn is_file(&self) -> bool {
        self.title.is_file()
    }

    /// Whether this page refers to a category
    pub fn is_category(&self) -> bool {
        self.title.is_category()
    }

    /// Load basic page information
    async fn info(&self) -> Result<&InfoResponseItem> {
        self.info
            .get_or_try_init(|| async {
                let mut resp: InfoResponse = self
                    .bot
                    .api
                    .query_response([("titles", self.title())])
                    .await?;
                Ok(resp
                    .query
                    .pages
                    .pop()
                    .expect("API response returned 0 pages"))
            })
            .await
    }

    /// Whether the page exists or not
    pub async fn exists(&self) -> Result<bool> {
        Ok(!self.info().await?.missing)
    }

    /// Get the canonical URL for this page
    pub async fn url(&self) -> Result<&str> {
        Ok(&self.info().await?.canonicalurl)
    }

    /// Whether the page is a redirect or not
    pub async fn is_redirect(&self) -> Result<bool> {
        Ok(self.info().await?.redirect)
    }

    /// The associated page for this page (subject page for a talk page or
    /// talk page for a subject page)
    pub async fn associated_page(&self) -> Result<Page> {
        self.bot.page(&self.info().await?.associatedpage)
    }

    /// If this page is a redirect, get the page it targets
    pub async fn redirect_target(&self) -> Result<Option<Page>> {
        // Optimize if we already know it's not a redirect
        if self.info.initialized() && !self.is_redirect().await? {
            return Ok(None);
        }
        // Do an API request to resolve the redirect
        let resp: InfoResponse = self
            .bot
            .api
            .query_response([("titles", self.title()), ("redirects", "1")])
            .await?;
        match resp.query.redirects.get(0) {
            Some(redirect) => Ok(Some(self.bot.page(&redirect.to)?)),
            None => Ok(None),
        }
    }

    fn get_baserevid(&self) -> Option<u64> {
        let mut baserevid = self.baserevid.lock().unwrap();
        let current = *baserevid;
        *baserevid = None;
        current
    }

    fn set_baserevid(&self, revid: u64) {
        let mut baserevid = self.baserevid.lock().unwrap();
        *baserevid = Some(revid)
    }

    /// Get Parsoid HTML for the latest revision of the page
    pub async fn html(&self) -> Result<ImmutableWikicode> {
        let resp = self.bot.parsoid.get(self.title()).await?;
        // Keep track of revision id for saving in the future
        if let Some(revid) = &resp.revision_id() {
            self.set_baserevid((*revid) as u64);
        }
        Ok(resp)
    }

    /// Get Parsoid HTML for the specified revision
    pub async fn revision_html(&self, revid: u32) -> Result<ImmutableWikicode> {
        Ok(self.bot.parsoid.get_revision(self.title(), revid).await?)
    }

    /// Get wikitext for the latest revision of the page
    pub async fn wikitext(&self) -> Result<String> {
        let resp = self
            .bot
            .api
            .get_value(&[
                ("action", "query"),
                ("titles", self.title()),
                ("prop", "revisions"),
                ("rvprop", "content|ids"),
                ("rvslots", "main"),
            ])
            .await?;
        let page = resp["query"]["pages"][0].as_object().unwrap();
        if page.contains_key("missing") {
            Err(Error::PageDoesNotExist(self.title().to_string()))
        } else {
            match page.get("revisions") {
                Some(revisions) => {
                    let revision = &revisions[0];
                    self.set_baserevid(revision["revid"].as_u64().unwrap());
                    Ok(revision["slots"]["main"]["content"]
                        .as_str()
                        .unwrap()
                        .to_string())
                }
                None => {
                    // Most likely invalid title, either way revision
                    // doesn't exist
                    Err(Error::PageDoesNotExist(self.title().to_string()))
                }
            }
        }
    }

    /// Save the page using the specified HTML
    pub async fn save<S: Into<Saveable>>(
        &self,
        edit: S,
        opts: &SaveOptions,
    ) -> Result<EditResponse> {
        let edit = edit.into();
        let wikitext = match edit {
            Saveable::Html(html) => {
                // Check {{nobots}}
                if self.bot.config.respect_nobots {
                    let username = self
                        .bot
                        .config
                        .username
                        .clone()
                        .unwrap_or_else(|| "unknown".to_string());
                    if !crate::utils::nobots(&html, &username)? {
                        return Err(Error::Nobots);
                    }
                }
                self.bot.parsoid.transform_to_wikitext(&html).await?
            }
            Saveable::Wikitext(wikitext) => {
                // TODO: check {{nobots}}
                wikitext
            }
        };
        let mut params: Vec<(&'static str, String)> = [
            ("action", "edit".to_string()),
            ("title", self.title().to_string()),
            ("text", wikitext),
            ("summary", opts.summary.to_string()),
        ]
        .to_vec();
        // Edit conflict detection
        if let Some(revid) = self.get_baserevid() {
            params.push(("baserevid", revid.to_string()));
        }
        // Even more basic edit conflict detection if we already have it
        if self.info.initialized() {
            if self.exists().await? {
                // Exists, don't create a new page
                params.push(("nocreate", "1".to_string()));
            } else {
                // Missing, only create a new page
                params.push(("createonly", "1".to_string()));
            }
        }
        if opts.mark_as_bot.unwrap_or(self.bot.config.mark_as_bot) {
            params.push(("bot", "1".to_string()));
        }
        if !opts.tags.is_empty() {
            params.push(("tags", opts.tags.join("|")));
        }
        // TODO: would be nice if we could output a sleep message here
        self.bot.state.save_timer.lock().await.tick().await;
        info!("Saving [[{}]]", self.title());
        let resp: Value = self.bot.api.post_with_token("csrf", &params).await?;
        match resp["edit"]["result"].as_str() {
            Some("Success") => {
                Ok(serde_json::from_value(resp["edit"].clone())?)
            }
            // Some legacy code might return "result": "Failure" but the
            // structure is totally unspecified, so we're best off just
            // passing the entire blob into the error in the hope it
            // contains some clue.
            _ => Err(Error::UnknownSaveFailure(resp)),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::tests::{is_authenticated, testwp};
    use crate::Error;
    use std::time::{Duration, SystemTime};

    #[tokio::test]
    async fn test_exists() {
        let bot = testwp().await;
        let page = bot.page("Main Page").unwrap();
        assert!(page.exists().await.unwrap());
        let page2 = bot.page("DoesNotExistPlease").unwrap();
        assert!(!page2.exists().await.unwrap());
    }

    #[tokio::test]
    async fn test_get_redirect_target() {
        let bot = testwp().await;
        let redir = bot.page("Redirect").unwrap();
        let target = redir.redirect_target().await.unwrap().unwrap();
        // "Redirect" points to "Main Page"
        assert_eq!(target.title(), "Main Page");
        // "Main Page" is not a redirect
        assert!(target.redirect_target().await.unwrap().is_none());
    }

    #[tokio::test]
    async fn test_get_content() {
        let bot = testwp().await;
        let page = bot.page("Main Page").unwrap();
        let html = page.html().await.unwrap().into_mutable();
        assert_eq!(html.title().unwrap(), "Main Page".to_string());
        assert_eq!(
            html.select_first("b").unwrap().text_contents(),
            "test wiki".to_string()
        );
        let wikitext = page.wikitext().await.unwrap();
        assert!(wikitext.contains("'''test wiki'''"));
    }

    #[tokio::test]
    async fn test_missing_page() {
        let bot = testwp().await;
        let page = bot.page("DoesNotExistPlease").unwrap();
        let err = page.html().await.unwrap_err();
        match err {
            Error::PageDoesNotExist(page) => {
                assert_eq!(&page, "DoesNotExistPlease")
            }
            err => {
                panic!("Unexpected error: {:?}", err)
            }
        }
        let err2 = page.wikitext().await.unwrap_err();
        match err2 {
            Error::PageDoesNotExist(page) => {
                assert_eq!(&page, "DoesNotExistPlease")
            }
            err => {
                panic!("Unexpected error: {:?}", err)
            }
        }
    }

    #[tokio::test]
    async fn test_save() {
        if !is_authenticated() {
            return;
        }

        let bot = testwp().await;
        let page = bot.page("mwbot-rs/Save").unwrap();
        let wikitext = format!(
            "It has been {} seconds since the epoch.",
            SystemTime::now()
                .duration_since(SystemTime::UNIX_EPOCH)
                .unwrap()
                .as_secs()
        );
        let mut retries = 0;
        loop {
            let resp = page
                .save(
                    wikitext.to_string(),
                    &SaveOptions::summary("Test suite edit"),
                )
                .await;
            match resp {
                Ok(resp) => {
                    assert_eq!(&resp.title, "Mwbot-rs/Save");
                    return;
                }
                Err(Error::EditConflict(_)) => {
                    if retries > 5 {
                        panic!("hit more than 5 edit conflicts");
                    }
                    retries += 1;
                    tokio::time::sleep(Duration::from_secs(5)).await;
                    continue;
                }
                Err(ref err) => {
                    dbg!(&resp);
                    panic!("{}", err);
                }
            }
        }
    }

    #[tokio::test]
    async fn test_protected() {
        if !is_authenticated() {
            return;
        }

        let bot = testwp().await;
        let page = bot.page("mwbot-rs/Protected").unwrap();
        let wikitext = "Wait, I can edit this page?".to_string();
        let error = page
            .save(wikitext, &SaveOptions::summary("Test suite edit"))
            .await
            .unwrap_err();
        dbg!(&error);
        assert!(matches!(error, Error::ProtectedPage));
    }

    #[tokio::test]
    async fn test_spamfilter() {
        if !is_authenticated() {
            return;
        }

        let bot = testwp().await;
        let page = bot.page("mwbot-rs/SpamBlacklist").unwrap();
        let wikitext = "https://bitly.com/12345".to_string();
        let error = page
            .save(wikitext, &SaveOptions::summary("Test suite edit"))
            .await
            .unwrap_err();
        if let Error::SpamFilter { matches, .. } = error {
            assert_eq!(matches, vec!["bitly.com".to_string()])
        } else {
            panic!("{:?} doesn't match", error)
        }
    }

    #[tokio::test]
    async fn test_partialblock() {
        if !is_authenticated() {
            return;
        }
        let bot = testwp().await;
        let page = bot.page("Mwbot-rs/Partially blocked").unwrap();
        let error = page
            .save(
                "I shouldn't be able to edit this".to_string(),
                &SaveOptions::summary("Test suite edit"),
            )
            .await
            .unwrap_err();
        if let Error::PartiallyBlocked { info, .. } = error {
            assert_eq!(info, "You have been blocked from editing this page.");
        } else {
            panic!("{:?} doesn't match", error);
        }
    }

    /// Regression test to verify we don't panic on invalid titles
    /// https://gitlab.com/mwbot-rs/mwbot/-/issues/33
    ///
    /// Mostly moot now that we have proper title validation
    #[tokio::test]
    async fn test_invalidtitle() {
        let bot = testwp().await;
        // Should return an error
        let err = bot.page("<invalid title>").unwrap_err();
        assert!(matches!(err, Error::InvalidTitle(_)));
        let err = bot.page("Special:BlankPage").unwrap_err();
        assert!(matches!(err, Error::InvalidPage));
    }

    #[tokio::test]
    async fn test_editconflict() {
        let bot = testwp().await;
        let page = bot.page("mwbot-rs/Save").unwrap();
        // Fake a older baserevid in
        page.set_baserevid(495829);
        let err = page
            .save(
                "This should fail",
                &SaveOptions::summary("this should fail"),
            )
            .await
            .unwrap_err();
        assert!(matches!(err, Error::EditConflict(_)));
    }

    #[tokio::test]
    async fn test_associated_page() {
        let bot = testwp().await;
        let page = bot.page("Main Page").unwrap();
        assert_eq!(
            page.associated_page().await.unwrap().title(),
            "Talk:Main Page"
        );
    }
}
