/*
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

use crate::edit::{EditResponse, SaveOptions, Saveable};
use crate::{Bot, Error, Result, Wikicode};
use log::info;
use mwapi_responses::prelude::*;
use parsoid::immutable::ImmutableWikicode;
use parsoid::Result as ParsoidResult;

/// Represents a wiki page and provides accessors and mutators.
#[derive(Debug, Clone)]
pub struct Page {
    pub(crate) bot: Bot,
    pub(crate) title: String,
}

// Use placeholder inprop for now
#[query(prop = "info", inprop = "associatedpage", redirects = "1")]
struct InfoResponse {}

impl Page {
    /// Get the title of the page
    pub fn title(&self) -> &str {
        &self.title
    }

    /// Whether the page exists or not
    pub async fn exists(&self) -> Result<bool> {
        let mut params = InfoResponse::params().to_vec();
        params.push(("titles", &self.title));
        let resp: InfoResponse = self.bot.api.get(&params).await?;
        Ok(!resp.query.pages[0].missing)
    }

    /// If this page is a redirect, get the page it targets
    pub async fn get_redirect_target(&self) -> Result<Option<Page>> {
        let mut params = InfoResponse::params().to_vec();
        params.push(("titles", &self.title));
        let resp: InfoResponse = self.bot.api.get(&params).await?;
        Ok(resp
            .query
            .redirects
            .get(0)
            .map(|redirect| self.bot.get_page(&redirect.to)))
    }

    /// Get Parsoid HTML for the latest revision of the page
    pub async fn get_html(&self) -> ParsoidResult<Wikicode> {
        Ok(self.bot.parsoid.get(&self.title).await?)
    }

    /// Get Parsoid HTML for the specified revision
    pub async fn get_revision_html(
        &self,
        revid: u32,
    ) -> ParsoidResult<Wikicode> {
        Ok(self.bot.parsoid.get_revision(&self.title, revid).await?)
    }

    pub async fn get_wikitext(&self) -> Result<String> {
        let resp = self
            .bot
            .api
            .get_value(&[
                ("action", "query"),
                ("titles", &self.title),
                ("prop", "revisions"),
                ("rvprop", "content"),
                ("rvslots", "main"),
            ])
            .await?;
        let page = resp["query"]["pages"][0].as_object().unwrap();
        if page.contains_key("missing") {
            Err(Error::PageDoesNotExist(self.title.to_string()))
        } else {
            match page.get("revisions") {
                Some(revisions) => Ok(revisions[0]["slots"]["main"]["content"]
                    .as_str()
                    .unwrap()
                    .to_string()),
                None => {
                    // Most likely invalid title, either way revision
                    // doesn't exist
                    Err(Error::PageDoesNotExist(self.title.to_string()))
                }
            }
        }
    }

    // Internal wrapper so `Wikicode` (non-Send) is scoped
    fn nobots(&self, html: &ImmutableWikicode, username: &str) -> Result<bool> {
        let code = html.clone().to_mutable();
        crate::utils::nobots(&code, username)
    }

    /// Save the page using the specified HTML
    pub async fn save<S: Into<Saveable>>(
        &self,
        edit: S,
        opts: &SaveOptions,
    ) -> Result<EditResponse> {
        let edit = edit.into();
        let wikitext = if let Some(html) = edit.html {
            // Check {{nobots}}
            if self.bot.config.respect_nobots {
                let username = self
                    .bot
                    .config
                    .username
                    .clone()
                    .unwrap_or_else(|| "unknown".to_string());
                if !self.nobots(&html, &username)? {
                    return Err(Error::Nobots);
                }
            }
            self.bot.parsoid.transform_to_wikitext(&html).await?
        } else if let Some(wikitext) = edit.wikitext {
            // TODO: check {{nobots}}
            wikitext
        } else {
            unreachable!("Not possible to construct a EditRequest with no HTML nor wikitext")
        };
        let mut params: Vec<(&'static str, String)> = [
            ("action", "edit".to_string()),
            ("title", self.title.to_string()),
            ("text", wikitext),
            ("summary", opts.summary.to_string()),
        ]
        .to_vec();
        if let Some(revid) = edit.baserevid {
            params.push(("baserevid", revid.to_string()));
        }
        if opts.mark_as_bot.unwrap_or(self.bot.config.mark_as_bot) {
            params.push(("bot", "1".to_string()));
        }
        if !opts.tags.is_empty() {
            params.push(("tags", opts.tags.join("|")));
        }
        // TODO: would be nice if we could output a sleep message here
        self.bot.state.save_timer.lock().await.tick().await;
        info!("Saving [[{}]]", &self.title);
        let resp = self.bot.api.post_with_token("csrf", &params).await?;
        match resp["edit"]["result"].as_str() {
            Some("Success") => {
                Ok(serde_json::from_value(resp["edit"].clone())?)
            }
            // Some legacy code might return "result": "Failure" but the
            // structure is totally unspecified, so we're best off just
            // passing the entire blob into the error in the hope it
            // contains some clue.
            _ => Err(Error::UnknownSaveFailure(resp)),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::tests::{is_authenticated, testwp};
    use crate::Error;
    use std::time::SystemTime;

    #[tokio::test]
    async fn test_exists() {
        let bot = testwp().await;
        let page = bot.get_page("Main Page");
        assert!(page.exists().await.unwrap());
        let page2 = bot.get_page("DoesNotExistPlease");
        assert!(!page2.exists().await.unwrap());
    }

    #[tokio::test]
    async fn test_get_redirect_target() {
        let bot = testwp().await;
        let redir = bot.get_page("Redirect");
        let target = redir.get_redirect_target().await.unwrap().unwrap();
        // "Redirect" points to "Main Page"
        assert_eq!(target.title(), "Main Page");
        // "Main Page" is not a redirect
        assert!(target.get_redirect_target().await.unwrap().is_none());
    }

    #[tokio::test]
    async fn test_get_content() {
        let bot = testwp().await;
        let page = bot.get_page("Main Page");
        let html = page.get_html().await.unwrap();
        assert_eq!(html.get_title().unwrap(), "Main Page".to_string());
        assert_eq!(
            html.select_first("b").unwrap().text_contents(),
            "test wiki".to_string()
        );
        let wikitext = page.get_wikitext().await.unwrap();
        assert!(wikitext.contains("'''test wiki'''"));
    }

    #[tokio::test]
    async fn test_missing_page() {
        let bot = testwp().await;
        let page = bot.get_page("DoesNotExistPlease");
        let err = page.get_html().await.unwrap_err();
        match err {
            Error::PageDoesNotExist(page) => {
                assert_eq!(&page, "DoesNotExistPlease")
            }
            err => {
                panic!("Unexpected error: {:?}", err)
            }
        }
        let err2 = page.get_wikitext().await.unwrap_err();
        match err2 {
            Error::PageDoesNotExist(page) => {
                assert_eq!(&page, "DoesNotExistPlease")
            }
            err => {
                panic!("Unexpected error: {:?}", err)
            }
        }
    }

    #[tokio::test]
    async fn test_save() {
        if !is_authenticated() {
            return;
        }

        let bot = testwp().await;
        let page = bot.get_page("mwbot-rs/Save");
        let wikitext = format!(
            "It has been {} seconds since the epoch.",
            SystemTime::now()
                .duration_since(SystemTime::UNIX_EPOCH)
                .unwrap()
                .as_secs()
        );
        let resp = page
            .save(wikitext, &SaveOptions::summary("Test suite edit"))
            .await
            .unwrap();
        dbg!(&resp);
        assert_eq!(&resp.title, "Mwbot-rs/Save");
    }

    #[tokio::test]
    async fn test_protected() {
        if !is_authenticated() {
            return;
        }

        let bot = testwp().await;
        let page = bot.get_page("mwbot-rs/Protected");
        let wikitext = "Wait, I can edit this page?".to_string();
        let error = page
            .save(wikitext, &SaveOptions::summary("Test suite edit"))
            .await
            .unwrap_err();
        dbg!(&error);
        assert!(matches!(error, Error::ProtectedPage));
    }

    #[tokio::test]
    async fn test_spamfilter() {
        if !is_authenticated() {
            return;
        }

        let bot = testwp().await;
        let page = bot.get_page("mwbot-rs/SpamBlacklist");
        let wikitext = "https://bitly.com/12345".to_string();
        let error = page
            .save(wikitext, &SaveOptions::summary("Test suite edit"))
            .await
            .unwrap_err();
        if let Error::SpamFilter { matches, .. } = error {
            assert_eq!(matches, vec!["bitly.com".to_string()])
        } else {
            panic!("{:?} doesn't match", error)
        }
    }

    #[tokio::test]
    async fn test_partialblock() {
        if !is_authenticated() {
            return;
        }
        let bot = testwp().await;
        let page = bot.get_page("Mwbot-rs/Partially blocked");
        let error = page
            .save(
                "I shouldn't be able to edit this".to_string(),
                &SaveOptions::summary("Test suite edit"),
            )
            .await
            .unwrap_err();
        if let Error::PartiallyBlocked { info, .. } = error {
            assert_eq!(info, "You have been blocked from editing this page.");
        } else {
            panic!("{:?} doesn't match", error);
        }
    }

    /// Regression test to verify we don't panic on invalid titles
    /// https://gitlab.com/mwbot-rs/mwbot/-/issues/33
    #[tokio::test]
    async fn test_invalidtitle() {
        let bot = testwp().await;
        let page = bot.get_page("<invalid title>");
        // Should return some error
        page.get_wikitext().await.unwrap_err();
        page.get_redirect_target().await.unwrap_err();
    }
}
