/*
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

use crate::{Bot, Page, Result};
use mwapi_responses::prelude::*;
use std::collections::HashMap;
use tokio::sync::mpsc;

#[query(list = "categorymembers", cmprop = "title", cmlimit = "max")]
struct CategoryResponse;

/// Get pages that are in the given category
pub fn categorymembers(bot: &Bot, title: &str) -> mpsc::Receiver<Result<Page>> {
    let (tx, rx) = mpsc::channel(50);
    let bot = bot.clone();
    let title = title.to_string();
    tokio::spawn(async move {
        let mut params: HashMap<String, String> = CategoryResponse::params()
            .iter()
            .map(|(k, v)| (k.to_string(), v.to_string()))
            .collect();
        params.insert("cmtitle".to_string(), title);
        let mut continue_: HashMap<String, String> = HashMap::new();
        loop {
            for (param, value) in &continue_ {
                params.insert(param.to_string(), value.to_string());
            }
            let resp: CategoryResponse = match bot.api.get(&params).await {
                Ok(resp) => resp,
                Err(err) => {
                    // TODO: is unwrap safe here?
                    tx.send(Err(err)).await.unwrap();
                    break;
                }
            };
            for item in resp.items() {
                // TODO: is unwrap safe here?
                tx.send(Ok(bot.get_page(&item.title))).await.unwrap();
            }
            if resp.continue_.is_empty() {
                // All done
                break;
            } else {
                continue_ = resp.continue_.clone();
            }
        }
    });
    rx
}

#[query(list = "embeddedin", eilimit = "max")]
struct EmbeddedInResponse;

/// Get pages that transclude the given template
pub fn embeddedin(bot: &Bot, title: &str) -> mpsc::Receiver<Result<Page>> {
    let (tx, rx) = mpsc::channel(50);
    let bot = bot.clone();
    let title = title.to_string();
    tokio::spawn(async move {
        let mut params: HashMap<String, String> = EmbeddedInResponse::params()
            .iter()
            .map(|(k, v)| (k.to_string(), v.to_string()))
            .collect();
        params.insert("eititle".to_string(), title);
        let mut continue_: HashMap<String, String> = HashMap::new();
        loop {
            for (param, value) in &continue_ {
                params.insert(param.to_string(), value.to_string());
            }
            let resp: EmbeddedInResponse = match bot.api.get(&params).await {
                Ok(resp) => resp,
                Err(err) => {
                    // TODO: is unwrap safe here?
                    tx.send(Err(err)).await.unwrap();
                    break;
                }
            };
            for item in resp.items() {
                // TODO: is unwrap safe here?
                tx.send(Ok(bot.get_page(&item.title))).await.unwrap();
            }
            if resp.continue_.is_empty() {
                // All done
                break;
            } else {
                continue_ = resp.continue_.clone();
            }
        }
    });
    rx
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::tests::testwp;

    #[tokio::test]
    async fn test_categorymembers() {
        let bot = testwp().await;
        let mut members = categorymembers(&bot, "Category:!Requests");
        while let Some(page) = members.recv().await {
            let page = page.unwrap();
            // This page is the last one, so it should require at least one continuation
            if page.title() == "Category:Unsuccessful requests for permissions"
            {
                assert!(true);
                return;
            }
        }

        panic!("Unable to find the page");
    }

    #[tokio::test]
    async fn test_embeddedin() {
        let bot = testwp().await;
        let mut embeddedin = embeddedin(&bot, "Template:1x");
        let mut count = 0;
        while let Some(page) = embeddedin.recv().await {
            page.unwrap();
            count += 1;
            if count == 5 {
                break;
            }
        }
        assert_eq!(count, 5);
    }
}
