/*
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

//! A MediaWiki Bot framework
//!
//! `mwbot` provides a batteries-included framework for building bots
//! for MediaWiki wikis. The goal is to provide a high-level API on top
//! of the [mwapi](https://docs.rs/mwapi) and
//! [parsoid](https://docs.rs/parsoid) crates.
//!
//! ## Configuration
//! Create a `mwbot.toml` file with the following structure:
//! ```toml
//! api_url = "https://en.wikipedia.org/w/api.php"
//! rest_url = "https://en.wikipedia.org/api/rest_v1"
//!
//! [auth]
//! username = "Example"
//! password = "a BotPassword"
//! ```
//!
//! Using `Bot::from_default_config()` will look in the current directory
//! for `mwbot.toml` before looking in the user's config directory. A
//! custom path can be specified by using `Bot::from_config(...)`.
//!
//! More to come.
use log::{debug, info};
pub use mwapi::Client as ApiClient;
use mwapi::ErrorFormat;
pub use mwapi_errors::Error;
use std::{path::Path, sync::Arc};
use tokio::{sync::Mutex, time};

mod config;
mod error;
mod page;
#[cfg(unix)]
mod unix;
mod utils;

pub type Result<T, E = Error> = std::result::Result<T, E>;
pub mod parsoid {
    pub use parsoid::prelude::*;
}

pub use error::ConfigError;
pub use page::Page;

use crate::parsoid::*;

/// Main bot class
#[derive(Clone, Debug)]
pub struct Bot {
    api: ApiClient,
    parsoid: ParsoidClient,
    state: BotState,
    config: Arc<BotConfig>,
}

/// Static, read-only settings
#[derive(Clone, Debug)]
struct BotConfig {
    // TODO: figure out something better than this
    username: Option<String>,
    mark_as_bot: bool,
}

/// Dynamic state
#[derive(Clone, Debug)]
struct BotState {
    save_timer: Arc<Mutex<time::Interval>>,
}

impl Bot {
    /// Load Bot configuration from a default location, first look at
    /// `mwbot.toml` in the current directory, otherwise look in the
    /// platform's config directory:
    ///
    /// * Linux: `$XDG_CONFIG_HOME` or `$HOME/.config`
    /// * macOS: `$HOME/Library/Application Support`
    /// * Windows: `{FOLDERID_RoamingAppData}`
    pub async fn from_default_config() -> Result<Self, ConfigError> {
        let path = {
            let first = std::path::Path::new("mwbot.toml");
            if first.exists() {
                first.to_path_buf()
            } else {
                dirs::config_dir()
                    .expect("Cannot find config directory")
                    .join("mwbot.toml")
            }
        };
        Self::from_path(&path).await
    }

    /// Load Bot configuration from the specified path.
    pub async fn from_path(path: &Path) -> Result<Self, ConfigError> {
        debug!("Reading config from {:?}", path);
        let config: config::Config =
            toml::from_str(&std::fs::read_to_string(path)?)?;
        // Check file permissions if there are credentials
        if config.auth.is_some() {
            check_file_permissions(path)?;
        }
        Self::from_config(config).await
    }

    async fn from_config(config: config::Config) -> Result<Self, ConfigError> {
        let mut api = ApiClient::builder(&config.api_url)
            .set_maxlag(5)
            .set_errorformat(ErrorFormat::Wikitext);
        let mut user_agent = vec![];
        // FIXME: do better
        let mut username = None;
        if let Some(auth) = config.auth {
            info!("Logging in as {}", &auth.username);
            api = api.set_botpassword(&auth.username, &auth.password);
            let normalized = normalize_username(&auth.username);
            user_agent.push(format!("User:{}", &normalized));
            username = Some(normalized);
        }
        user_agent.push(format!("mwbot-rs/{}", env!("CARGO_PKG_VERSION")));
        let user_agent = user_agent.join(" ");
        let mut interval = time::interval(time::Duration::from_secs(
            config.edit.save_delay.unwrap_or(10),
        ));
        interval.set_missed_tick_behavior(time::MissedTickBehavior::Delay);
        let api = api.set_user_agent(&user_agent).build().await?;
        let http = api.http_client().clone();
        Ok(Self {
            api,
            parsoid: ParsoidClient::new_with_client(&config.rest_url, http),
            config: Arc::new(BotConfig {
                username,
                mark_as_bot: config.edit.mark_as_bot.unwrap_or(true),
            }),
            state: BotState {
                save_timer: Arc::new(Mutex::new(interval)),
            },
        })
    }

    /// Get a reference to the underlying [`mwapi::Client`](https://docs.rs/mwapi/latest/mwapi/struct.Client.html)
    /// to make arbitrary API requests
    pub fn get_api(&self) -> &ApiClient {
        &self.api
    }

    /// Get a reference to the underlying [`parsoid::Client`](https://docs.rs/parsoid/latest/parsoid/struct.Client.html)
    /// to make arbitrary Parsoid API requests
    pub fn get_parsoid(&self) -> &ParsoidClient {
        &self.parsoid
    }

    /// Get a `Page` on this wiki. The specified title should be a full title,
    /// including namespace. Currently, this does no validation on the
    /// provided input.
    pub fn get_page(&self, title: &str) -> Page {
        Page {
            bot: self.clone(),
            title: title.to_string(),
        }
    }
}

/// Verify file permissions are not obviously misconfigured:
/// * If the file is owned by the current user, it should only be readable to
///   that user
/// * If the file is owned by another user, it should not be world readable
///
/// TODO: support [extended ACLs](https://wiki.archlinux.org/title/Access_Control_Lists)
#[cfg(unix)]
fn check_file_permissions(path: &Path) -> Result<(), ConfigError> {
    use std::os::unix::fs::{MetadataExt, PermissionsExt};
    let metadata = std::fs::metadata(path)?;
    let mode = metadata.permissions().mode();
    let owner = metadata.uid();
    let current_user = unix::get_current_uid();
    let group_readable = mode & 0o40 != 0;
    let world_readable = mode & 0o4 != 0;
    if owner == current_user {
        // We own the file, so it should be only readable to us
        if group_readable || world_readable {
            Err(ConfigError::ReadableConfig(mode))
        } else {
            Ok(())
        }
    } else {
        // Someone else owns the file, so just check it's not world readable
        if world_readable {
            Err(ConfigError::WorldReadableConfig(mode))
        } else {
            Ok(())
        }
    }
}

#[cfg(not(unix))]
fn check_file_permissions(_path: &Path) -> Result<(), ConfigError> {
    // XXX: Implement this for other platforms
    Ok(())
}

fn normalize_username(original: &str) -> String {
    // MW normalization, underscores to spaces
    let name = original.replace("_", " ");
    // If it's a bot password, strip the @<name> part
    match name.split_once('@') {
        Some((name, _)) => name.to_string(),
        None => name,
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    pub(crate) async fn testwp() -> Bot {
        Bot::from_config(config::Config {
            api_url: "https://test.wikipedia.org/w/api.php".to_string(),
            rest_url: "https://test.wikipedia.org/api/rest_v1".to_string(),
            auth: None,
            edit: Default::default(),
        })
        .await
        .unwrap()
    }

    // TODO: use this
    pub(crate) async fn _testwp_authed() -> Option<Bot> {
        let username = std::env::var("MWAPI_USERNAME");
        let password = std::env::var("MWAPI_PASSWORD");
        let auth = if username.is_err() || password.is_err() {
            return None;
        } else {
            Some(config::Auth {
                username: username.unwrap(),
                password: password.unwrap(),
            })
        };

        Bot::from_config(config::Config {
            api_url: "https://test.wikipedia.org/w/api.php".to_string(),
            rest_url: "https://test.wikipedia.org/api/rest_v1".to_string(),
            auth,
            edit: Default::default(),
        })
        .await
        .ok()
    }

    #[test]
    fn test_normalize_username() {
        assert_eq!(&normalize_username("Foo"), "Foo");
        assert_eq!(&normalize_username("Foo_bar"), "Foo bar");
        assert_eq!(&normalize_username("Foo@bar"), "Foo");
    }

    #[tokio::test]
    async fn test_get_api() {
        let bot = testwp().await;
        // No errors
        bot.get_api()
            .get_value(&[("action", "query")])
            .await
            .unwrap();
    }

    #[tokio::test]
    async fn test_get_page() {
        let bot = testwp().await;
        let page = bot.get_page("Example");
        assert_eq!(page.title(), "Example");
    }

    #[tokio::test]
    async fn test_user_agent() {
        let bot = testwp().await;
        let version = env!("CARGO_PKG_VERSION");
        let resp: serde_json::Value = bot
            .get_api()
            .http_client()
            .get("https://httpbin.org/user-agent")
            .send()
            .await
            .unwrap()
            .json()
            .await
            .unwrap();
        let user_agent = resp["user-agent"].as_str().unwrap();
        match &bot.config.username {
            Some(username) => {
                assert_eq!(
                    user_agent,
                    &format!("User:{} mwbot-rs/{}", username, version)
                );
            }
            None => {
                assert_eq!(user_agent, &format!("mwbot-rs/{}", version));
            }
        }
    }
}
