/*
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
use crate::Result;
use parsoid::{Wikicode, WikinodeIterator};

/// Implementation of the `{{nobots}}` exclusion mechanism as documented
/// at [Template:Bots](https://en.wikipedia.org/wiki/Template:Bots).
///
/// If true, the page is fine to edit. If false, don't edit.
pub fn nobots(html: &Wikicode, user: &str) -> Result<bool> {
    for temp in html.filter_templates()? {
        if temp.name() == "Template:Bots" {
            for (param, value) in temp.get_params() {
                let bots: Vec<_> = value.split(' ').collect();
                if param == "allow" {
                    if value == "none" {
                        return Ok(false);
                    }
                    if bots.contains(&"all") || bots.contains(&user) {
                        return Ok(true);
                    }
                } else if param == "deny" {
                    if value == "none" {
                        return Ok(true);
                    }
                    if bots.contains(&"all") || bots.contains(&user) {
                        return Ok(false);
                    }
                }
            }

            if temp.raw_name() == "nobots" && temp.get_params().is_empty() {
                return Ok(false);
            }
        }
    }
    Ok(true)
}
