/*
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

use crate::{Bot, Error, Result, Wikicode};
use log::info;
use parsoid::Result as ParsoidResult;

/// Represents a wiki page and provides accessors and mutators.
#[derive(Debug, Clone)]
pub struct Page {
    pub(crate) bot: Bot,
    pub(crate) title: String,
}

#[derive(Default)]
pub struct EditRequest {
    wikitext: Option<String>,
    html: Option<String>,
    etag: Option<String>,
    baserevid: Option<u32>,
    summary: String,
    mark_as_bot: Option<bool>,
}

impl EditRequest {
    pub fn set_summary(mut self, summary: &str) -> Self {
        self.summary = summary.to_string();
        self
    }

    pub fn mark_as_bot(mut self, mark: bool) -> Self {
        self.mark_as_bot = Some(mark);
        self
    }
}

impl Page {
    /// Get the title of the page
    pub fn title(&self) -> &str {
        &self.title
    }

    /// Whether the page exists or not
    pub async fn exists(&self) -> Result<bool> {
        let resp = self
            .bot
            .api
            .get_value(&[("action", "query"), ("titles", &self.title)])
            .await?;
        Ok(!resp["query"]["pages"][0]["missing"]
            .as_bool()
            .unwrap_or_default())
    }

    /// Get Parsoid HTML for the latest revision of the page
    pub async fn get_html(&self) -> ParsoidResult<Wikicode> {
        Ok(self.bot.parsoid.get(&self.title).await?)
    }

    /// Get Parsoid HTML for the specified revision
    pub async fn get_revision_html(
        &self,
        revid: u32,
    ) -> ParsoidResult<Wikicode> {
        Ok(self.bot.parsoid.get_revision(&self.title, revid).await?)
    }

    pub async fn get_wikitext(&self) -> Result<String> {
        let resp = self
            .bot
            .api
            .get_value(&[
                ("action", "query"),
                ("titles", &self.title),
                ("prop", "revisions"),
                ("rvprop", "content"),
                ("rvslots", "main"),
            ])
            .await?;
        let page = resp["query"]["pages"][0].as_object().unwrap();
        if page.contains_key("missing") {
            Err(Error::PageDoesNotExist(self.title.to_string()))
        } else {
            let revisions = page.get("revisions").unwrap().clone();
            Ok(revisions[0]["slots"]["main"]["content"]
                .as_str()
                .unwrap()
                .to_string())
        }
    }

    pub fn prep_saving_html(&self, html: &Wikicode) -> Result<EditRequest> {
        let username = self
            .bot
            .config
            .username
            .clone()
            .unwrap_or_else(|| "unknown".to_string());
        if !crate::utils::nobots(html, &username)? {
            return Err(Error::ProtectedPage(
                self.title.to_string(),
                "{{nobots}}".to_string(),
            ));
        }
        Ok(EditRequest {
            html: Some(html.to_string()),
            etag: html.get_etag().map(|etag| etag.to_string()),
            baserevid: html.revision_id(),
            ..Default::default()
        })
    }

    pub fn prep_saving_wikitext(&self, wikitext: &str) -> Result<EditRequest> {
        // TODO: can we keep track of baserevid here?
        // TODO: implement {{nobots}}
        Ok(EditRequest {
            wikitext: Some(wikitext.to_string()),
            ..Default::default()
        })
    }

    /// Save the page using the specified HTML
    pub async fn save(&self, edit: EditRequest) -> Result<()> {
        let wikitext = if let Some(html) = edit.html {
            self.bot
                .parsoid
                .transform_to_wikitext_raw(
                    &html,
                    Some(self.title()),
                    edit.baserevid,
                    edit.etag.as_deref(),
                )
                .await?
        } else if let Some(wikitext) = edit.wikitext {
            wikitext
        } else {
            unreachable!("Not possible to construct a EditRequest with no HTML nor wikitext")
        };
        let mut params: Vec<(String, String)> = [
            ("action".to_string(), "edit".to_string()),
            ("title".to_string(), self.title.to_string()),
            ("text".to_string(), wikitext),
            ("summary".to_string(), edit.summary),
        ]
        .to_vec();
        if let Some(revid) = edit.baserevid {
            params.push(("baserevid".to_string(), revid.to_string()));
        }
        if !edit.mark_as_bot.unwrap_or(self.bot.config.mark_as_bot) {
            params.push(("bot".to_string(), "1".to_string()));
        }
        // TODO: would be nice if we could output a sleep message here
        self.bot.state.save_timer.lock().await.tick().await;
        info!("Saving [[{}]]", &self.title);
        self.bot.api.post_with_token("csrf", &params).await?;

        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use crate::tests::testwp;
    use crate::Error;

    #[tokio::test]
    async fn test_exists() {
        let bot = testwp().await;
        let page = bot.get_page("Main Page");
        assert!(page.exists().await.unwrap());
        let page2 = bot.get_page("DoesNotExistPlease");
        assert!(!page2.exists().await.unwrap());
    }

    #[tokio::test]
    async fn test_get_content() {
        let bot = testwp().await;
        let page = bot.get_page("Main Page");
        let html = page.get_html().await.unwrap();
        assert_eq!(html.title().unwrap(), "Main Page".to_string());
        assert_eq!(
            html.select_first("b").unwrap().text_contents(),
            "test wiki".to_string()
        );
        let wikitext = page.get_wikitext().await.unwrap();
        assert!(wikitext.contains("'''test wiki'''"));
    }

    #[tokio::test]
    async fn test_missing_page() {
        let bot = testwp().await;
        let page = bot.get_page("DoesNotExistPlease");
        let err = page.get_html().await.unwrap_err();
        match err {
            Error::PageDoesNotExist(page) => {
                assert_eq!(&page, "DoesNotExistPlease")
            }
            err => {
                panic!("Unexpected error: {:?}", err)
            }
        }
        let err2 = page.get_wikitext().await.unwrap_err();
        match err2 {
            Error::PageDoesNotExist(page) => {
                assert_eq!(&page, "DoesNotExistPlease")
            }
            err => {
                panic!("Unexpected error: {:?}", err)
            }
        }
    }
}
