/*
Copyright (C) 2021 Kunal Mehta <legoktm@member.fsf.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

use crate::{Bot, Error, Result, Wikicode};
use log::info;
use parsoid::Result as ParsoidResult;

/// Represents a wiki page and provides accessors and mutators.
pub struct Page<'a> {
    bot: &'a Bot,
    title: String,
}

impl Page<'_> {
    /// Create a new wiki page. Calling `Bot::get_page(...)` is probably
    /// better than doing this.
    pub fn new<'a>(bot: &'a Bot, title: &str) -> Page<'a> {
        Page {
            bot,
            title: title.to_string(),
        }
    }

    /// Get the title of the page
    pub fn title(&self) -> &str {
        &self.title
    }

    /// Whether the page exists or not
    pub async fn exists(&self) -> Result<bool> {
        let resp = self
            .bot
            .api
            .get(&[("action", "query"), ("titles", &self.title)])
            .await?;
        Ok(!resp["query"]["pages"][0]["missing"]
            .as_bool()
            .unwrap_or_default())
    }

    /// Get Parsoid HTML for the latest revision of the page
    pub async fn get_html(&self) -> ParsoidResult<Wikicode> {
        Ok(self.bot.parsoid.get(&self.title).await?)
    }

    /// Get Parsoid HTML for the specified revision
    pub async fn get_revision_html(
        &self,
        revid: u32,
    ) -> ParsoidResult<Wikicode> {
        Ok(self.bot.parsoid.get_revision(&self.title, revid).await?)
    }

    pub async fn get_wikitext(&self) -> Result<String> {
        let resp = self
            .bot
            .api
            .get(&[
                ("action", "query"),
                ("titles", &self.title),
                ("prop", "revisions"),
                ("rvprop", "content"),
                ("rvslots", "main"),
            ])
            .await?;
        let page = resp["query"]["pages"][0].as_object().unwrap();
        if page.contains_key("missing") {
            Err(Error::PageDoesNotExist(self.title.to_string()))
        } else {
            let revisions = page.get("revisions").unwrap().clone();
            Ok(revisions[0]["slots"]["main"]["content"]
                .as_str()
                .unwrap()
                .to_string())
        }
    }

    /// Save the page using the specified HTML
    pub async fn save_html(
        &self,
        html: &Wikicode,
        summary: &str,
    ) -> Result<()> {
        let wikitext = self.bot.parsoid.transform_to_wikitext(html).await?;
        let username = self
            .bot
            .username
            .clone()
            .unwrap_or_else(|| "unknown".to_string());
        if !crate::utils::nobots(html, &username)? {
            return Err(Error::ProtectedPage(
                self.title.to_string(),
                "{{nobots}}".to_string(),
            ));
        }
        let mut params: Vec<(String, String)> = [
            ("action".to_string(), "edit".to_string()),
            ("title".to_string(), self.title.to_string()),
            ("text".to_string(), wikitext),
            // TODO: make bot=1 conditional?
            ("bot".to_string(), "1".to_string()),
            ("summary".to_string(), summary.to_string()),
        ]
        .to_vec();
        if let Some(revid) = html.revision_id() {
            params.push(("baserevid".to_string(), revid.to_string()));
        }
        info!("Saving [[{}]]", &self.title);
        self.bot.api.post_with_token("csrf", &params).await?;

        Ok(())
    }

    /// Save the page using the specified wikitext
    pub async fn save_wikitext(
        &self,
        wikitext: &str,
        summary: &str,
    ) -> Result<()> {
        // TODO: implement {{nobots}} for wikitext
        // TODO: baserevid for edit conflict detection?
        self.bot
            .api
            .post_with_token(
                "csrf",
                &[
                    ("action", "edit"),
                    ("title", &self.title),
                    ("text", wikitext),
                    ("bot", "1"),
                    ("summary", summary),
                ],
            )
            .await?;
        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use crate::tests::testwp;
    use crate::Error;

    #[tokio::test]
    async fn test_exists() {
        let bot = testwp().await;
        let page = bot.get_page("Main Page");
        assert!(page.exists().await.unwrap());
        let page2 = bot.get_page("DoesNotExistPlease");
        assert!(!page2.exists().await.unwrap());
    }

    #[tokio::test]
    async fn test_get_content() {
        let bot = testwp().await;
        let page = bot.get_page("Main Page");
        let html = page.get_html().await.unwrap();
        assert_eq!(html.title().unwrap(), "Main Page".to_string());
        assert_eq!(
            html.select_first("b").unwrap().text_contents(),
            "test wiki".to_string()
        );
        let wikitext = page.get_wikitext().await.unwrap();
        assert!(wikitext.contains("'''test wiki'''"));
    }

    #[tokio::test]
    async fn test_missing_page() {
        let bot = testwp().await;
        let page = bot.get_page("DoesNotExistPlease");
        let err = page.get_html().await.unwrap_err();
        match err {
            Error::PageDoesNotExist(page) => {
                assert_eq!(&page, "DoesNotExistPlease")
            }
            err => {
                panic!("Unexpected error: {:?}", err)
            }
        }
        let err2 = page.get_wikitext().await.unwrap_err();
        match err2 {
            Error::PageDoesNotExist(page) => {
                assert_eq!(&page, "DoesNotExistPlease")
            }
            err => {
                panic!("Unexpected error: {:?}", err)
            }
        }
    }
}
