/*
Copyright (C) 2021 Kunal Mehta <legoktm@member.fsf.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

//! A MediaWiki Bot framework
//!
//! `mwbot` provides a batteries-included framework for building bots
//! for MediaWiki wikis. The goal is to provide a high-level API on top
//! of the [mwapi](https://docs.rs/mwapi) and
//! [parsoid](https://docs.rs/parsoid) crates.
//!
//! ## Configuration
//! Create a `mwbot.toml` file with the following structure:
//! ```toml
//! api_url = "https://en.wikipedia.org/w/api.php"
//! rest_url = "https://en.wikipedia.org/api/rest_v1"
//!
//! [auth]
//! username = "Example"
//! password = "a BotPassword"
//! ```
//!
//! Using `Bot::from_default_config()` will look in the current directory
//! for `mwbot.toml` before looking in the user's config directory. A
//! custom path can be specified by using `Bot::from_config(...)`.
//!
//! More to come.
use log::{debug, info};
pub use mwapi::Client as ApiClient;
use mwapi::ErrorFormat;
pub use mwapi_errors::Error;
use serde::Deserialize;
use std::path::Path;

mod config;
mod page;
mod utils;

pub type Result<T, E = Error> = std::result::Result<T, E>;
pub mod parsoid {
    pub use parsoid::prelude::*;
}

pub use config::ConfigError;
pub use page::Page;

use crate::parsoid::*;

/// Config file
#[derive(Deserialize)]
struct Config {
    api_url: String,
    rest_url: String,
    auth: Option<Auth>,
}

/// Login information (BotPassword)
#[derive(Deserialize)]
struct Auth {
    username: String,
    password: String,
}

/// Main bot class
pub struct Bot {
    api: ApiClient,
    parsoid: ParsoidClient,
    // TODO: figure out something better than this
    username: Option<String>,
}

impl Bot {
    /// Load Bot configuration from a default location, first look at
    /// `mwbot.toml` in the current directory, otherwise look in the
    /// platform's config directory:
    ///
    /// * Linux: `$XDG_CONFIG_HOME` or `$HOME/.config`
    /// * macOS: `$HOME/Library/Application Support`
    /// * Windows: `{FOLDERID_RoamingAppData}`
    pub async fn from_default_config() -> Result<Self, ConfigError> {
        let path = {
            let first = std::path::Path::new("mwbot.toml");
            if first.exists() {
                first.to_path_buf()
            } else {
                dirs::config_dir()
                    .expect("Cannot find config directory")
                    .join("mwbot.toml")
            }
        };
        Self::from_path(&path).await
    }

    /// Load Bot configuration from the specified path.
    pub async fn from_path(path: &Path) -> Result<Self, ConfigError> {
        debug!("Reading config from {:?}", path);
        let config: Config = toml::from_str(&std::fs::read_to_string(path)?)?;
        Self::from_config(config).await
    }

    async fn from_config(config: Config) -> Result<Self, ConfigError> {
        let mut api = ApiClient::builder(&config.api_url)
            .set_maxlag(5)
            .set_errorformat(ErrorFormat::Wikitext);
        let mut user_agent = vec![];
        // FIXME: do better
        let mut username = None;
        if let Some(auth) = config.auth {
            info!("Logging in as {}", &auth.username);
            api = api.set_botpassword(&auth.username, &auth.password);
            user_agent.push(format!("User:{}", &auth.username));
            username = Some(auth.username);
        }
        user_agent.push(format!("mwbot-rs/{}", env!("CARGO_PKG_VERSION")));
        let user_agent = user_agent.join(" ");
        Ok(Self {
            api: api.set_user_agent(&user_agent).build().await?,
            parsoid: ParsoidClient::new(&config.rest_url, &user_agent)?,
            username,
        })
    }

    /// Get a reference to the underlying [`mwapi::Client`](https://docs.rs/mwapi/latest/mwapi/struct.Client.html)
    /// to make arbitrary API requests
    pub fn get_api(&self) -> &ApiClient {
        &self.api
    }

    /// Get a reference to the underlying [`parsoid::Client`](https://docs.rs/parsoid/latest/parsoid/struct.Client.html)
    /// to make arbitrary Parsoid API requests
    pub fn get_parsoid(&self) -> &ParsoidClient {
        &self.parsoid
    }

    /// Get a `Page` on this wiki. The specified title should be a full title,
    /// including namespace. Currently, this does no validation on the
    /// provided input.
    pub fn get_page(&self, title: &str) -> Page {
        Page::new(self, title)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    pub(crate) async fn testwp() -> Bot {
        Bot::from_config(Config {
            api_url: "https://test.wikipedia.org/w/api.php".to_string(),
            rest_url: "https://test.wikipedia.org/api/rest_v1".to_string(),
            auth: None,
        })
        .await
        .unwrap()
    }

    #[tokio::test]
    async fn test_get_api() {
        let bot = testwp().await;
        // No errors
        bot.get_api().get(&[("action", "query")]).await.unwrap();
    }

    #[tokio::test]
    async fn test_get_page() {
        let bot = testwp().await;
        let page = bot.get_page("Example");
        assert_eq!(page.title(), "Example");
    }
}
