/*
Copyright (C) 2021 Kunal Mehta <legoktm@member.fsf.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

//! A MediaWiki Bot framework
//!
//! `mwbot` provides a batteries-included framework for building bots
//! for MediaWiki wikis. The goal is to provide a high-level API on top
//! of the [mwapi](https://docs.rs/mwapi) and
//! [parsoid](https://docs.rs/parsoid) crates.
//!
//! ## Configuration
//! Create a `mwbot.toml` file with the following structure:
//! ```toml
//! api_url = "https://en.wikipedia.org/w/api.php"
//! rest_url = "https://en.wikipedia.org/api/rest_v1"
//!
//! [auth]
//! username = "Example"
//! password = "a BotPassword"
//! ```
//!
//! Using `Bot::from_default_config()` will look in the current directory
//! for `mwbot.toml` before looking in the user's config directory. A
//! custom path can be specified by using `Bot::from_config(...)`.
//!
//! More to come.
use anyhow::Result;
use log::{debug, info};
pub use mwapi::Client as ApiClient;
use serde::Deserialize;
use std::path::Path;

pub mod parsoid {
    pub use parsoid::prelude::*;
}

use crate::parsoid::*;

/// Config file
#[derive(Deserialize)]
struct Config {
    api_url: String,
    rest_url: String,
    auth: Auth,
}

/// Login information (BotPassword)
#[derive(Deserialize)]
struct Auth {
    username: String,
    password: String,
}

/// Main bot class
pub struct Bot {
    api: ApiClient,
    parsoid: ParsoidClient,
}

impl Bot {
    /// Create a `Bot` instance from already created ApiClient/ParsoidClients
    pub fn new(api: ApiClient, parsoid: ParsoidClient) -> Self {
        Self { api, parsoid }
    }

    /// Load Bot configuration from a default location, first look at
    /// `mwbot.toml` in the current directory, otherwise look in the
    /// platform's config directory:
    ///
    /// * Linux: `$XDG_CONFIG_HOME` or `$HOME/.config`
    /// * macOS: `$HOME/Library/Application Support`
    /// * Windows: `{FOLDERID_RoamingAppData}`
    pub async fn from_default_config() -> Result<Self> {
        let path = {
            let first = std::path::Path::new("mwbot.toml");
            if first.exists() {
                first.to_path_buf()
            } else {
                dirs::config_dir()
                    .expect("Cannot find config directory")
                    .join("mwbot.toml")
            }
        };
        Ok(Self::from_config(&path).await?)
    }

    /// Load Bot configuration from the specified path.
    pub async fn from_config(path: &Path) -> Result<Self> {
        debug!("Reading config from {:?}", path);
        let config: Config = toml::from_str(&std::fs::read_to_string(path)?)?;
        info!("Logging in as {}", config.auth.username);
        let user_agent = format!(
            "User:{} mwbot-rs/{}",
            &config.auth.username,
            env!("CARGO_PKG_VERSION")
        );
        Ok(Self {
            api: ApiClient::bot_builder(&config.api_url)
                .set_user_agent(&user_agent)
                .set_botpassword(&config.auth.username, &config.auth.password)
                .build()
                .await?,
            parsoid: ParsoidClient::new(&config.rest_url, &user_agent)?,
        })
    }

    /// Get a reference to the underlying [`mwapi::Client`](https://docs.rs/mwapi/latest/mwapi/struct.Client.html)
    /// to make arbitrary API requests
    pub fn get_api(&self) -> &ApiClient {
        &self.api
    }

    /// Get a `Page` on this wiki. The specified title should be a full title,
    /// including namespace. Currently, this does no validation on the
    /// provided input.
    pub fn get_page(&self, title: &str) -> Page {
        Page::new(self, title)
    }
}

/// Represents a wiki page and provides accessors and mutators.
pub struct Page<'a> {
    bot: &'a Bot,
    title: String,
}

impl Page<'_> {
    /// Create a new wiki page. Calling `Bot::get_page(...)` is probably
    /// better than doing this.
    pub fn new<'a>(bot: &'a Bot, title: &str) -> Page<'a> {
        Page {
            bot,
            title: title.to_string(),
        }
    }

    /// Get Parsoid HTML for the latest revision of the page
    pub async fn get_html(&self) -> Result<Wikicode> {
        Ok(self.bot.parsoid.get(&self.title).await?)
    }

    /// Get Parsoid HTML for the specified revision
    pub async fn get_revision_html(&self, revid: u32) -> Result<Wikicode> {
        Ok(self.bot.parsoid.get_revision(&self.title, revid).await?)
    }

    /// Save the page using the specified HTML
    pub async fn save_html(
        &self,
        html: &Wikicode,
        summary: &str,
    ) -> Result<()> {
        let wikitext = self.bot.parsoid.transform_to_wikitext(html).await?;
        let mut params: Vec<(String, String)> = [
            ("action".to_string(), "edit".to_string()),
            ("title".to_string(), self.title.to_string()),
            ("text".to_string(), wikitext),
            // TODO: make bot=1 conditional?
            ("bot".to_string(), "1".to_string()),
            ("summary".to_string(), summary.to_string()),
        ]
        .to_vec();
        if let Some(revid) = html.revision_id() {
            params.push(("baserevid".to_string(), revid.to_string()));
        }
        info!("Saving [[{}]]", &self.title);
        self.bot.api.post_with_token("csrf", &params).await?;

        Ok(())
    }
}
