/*
Copyright (C) 2020 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

use proc_macro2::TokenStream as TokenStream2;
use quote::format_ident;
use quote::{quote, ToTokens};
use serde::Deserialize;
use std::fs;

#[derive(Debug, Clone, Deserialize)]
pub struct Metadata {
    pub name: String,
    pub mode: String,
    pub fieldname: String,
    pub prop: String,
    pub fields: Vec<Field>,
}

impl Metadata {
    pub fn new(name: &str) -> Self {
        // XXX: Is loading JSON files the best way to store/access this metadata?
        let path = format!("{}/data/{}.json", env!("CARGO_MANIFEST_DIR"), name);
        let contents = fs::read_to_string(path).unwrap_or_else(|_| {
            panic!("Unable to load JSON metadata: {}", name)
        });
        serde_json::from_str(&contents).unwrap_or_else(|_| {
            panic!("Unable to parse JSON metadata: {}", name)
        })
    }

    pub fn get_field(&self, prop: &str) -> &Field {
        for field in &self.fields {
            if field.prop == prop {
                return field;
            }
        }

        panic!("Could not find the field for prop={}", prop);
    }
}

#[derive(Debug, Clone, Deserialize)]
pub struct Field {
    /// Name of field in API response
    pub name: String,
    /// Rust type to map to
    pub type_: String,
    /// prop= value that gives this field
    pub prop: String,
    /// Value this field should be renamed to (e.g. "type" -> "type_")
    pub rename: Option<String>,
    /// Whether `#[serde(default)]` should be set on this field
    #[serde(default)]
    pub default: bool,
}

impl Field {
    fn get_fieldname(&self) -> String {
        match &self.rename {
            Some(name) => name.to_string(),
            None => self.name.to_string(),
        }
    }
}

impl ToTokens for Field {
    /// A field line looks like:
    /// ```ignore
    /// #[serde(default)] #[serde(rename = "foo")] foo = String;
    /// ```
    /// The serde attributes are optional.
    ///
    fn to_tokens(&self, tokens: &mut TokenStream2) {
        let name = format_ident!("{}", self.get_fieldname());
        let type_ = format_ident!("{}", self.type_);
        let default = if self.default {
            Some(quote! {
                #[serde(default)]
            })
        } else {
            None
        };
        let rename = if self.rename.is_some() {
            let name = self.name.to_string();
            Some(quote! {
                #[serde(rename = #name)]
            })
        } else {
            None
        };
        let stream = quote! {
            #default #rename pub #name: #type_,
        };

        stream.to_tokens(tokens);
    }
}
