/*
Copyright (C) 2020-2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

mod metadata;

extern crate proc_macro;
#[macro_use]
extern crate syn;

use proc_macro::TokenStream;
use proc_macro2::TokenStream as TokenStream2;
use quote::{format_ident, quote, ToTokens};
use std::collections::HashMap;
use syn::{AttributeArgs, DeriveInput, Ident, NestedMeta};

#[derive(Debug, Clone)]
struct Params {
    params: HashMap<String, String>,
}

impl Params {
    fn new() -> Self {
        Self {
            params: HashMap::new(),
        }
    }

    fn set_defaults(&mut self) {
        if !self.params.contains_key("action") {
            // Default to action=query
            self.insert("action".to_string(), "query".to_string());
        }

        // Force format=json&formatversion=2
        self.insert("format".to_string(), "json".to_string());
        self.insert("formatversion".to_string(), "2".to_string());
    }

    fn action(&self) -> String {
        self.params
            .get("action")
            .expect("No action parameter set")
            .to_string()
    }

    // Proxy for convenience
    fn insert(&mut self, key: String, val: String) {
        self.params.insert(key, val);
    }

    // Proxy for convenience
    fn is_empty(&self) -> bool {
        self.params.is_empty()
    }
}

#[proc_macro_attribute]
pub fn query(args: TokenStream, input: TokenStream) -> TokenStream {
    // Construct a representation of Rust code as a syntax tree
    // that we can manipulate
    let args = parse_macro_input!(args as AttributeArgs);
    let params = process_args(&args);
    let input = parse_macro_input!(input as DeriveInput);

    // Build the implementation
    impl_query(&input, &params).into()
}

fn process_args(args: &[NestedMeta]) -> Params {
    let mut params = Params::new();
    for arg in args {
        if let NestedMeta::Meta(syn::Meta::NameValue(meta)) = arg {
            let name = meta.path.get_ident().unwrap().to_string();
            let val = match &meta.lit {
                syn::Lit::Str(val) => val.value(),
                _ => panic!("Non-string value found for parameter '{}'", &name),
            };
            params.insert(name, val);
        } else {
            panic!("Unexpected value");
        }
    }
    if params.is_empty() {
        panic!("No parameters provided");
    }

    params.set_defaults();
    params
}

/// Extract the list of modules to be used from
/// the parameters so we can look up their response
/// structures.
fn get_modules(params: &Params) -> Vec<String> {
    let action = params.action();
    if action != "query" {
        // TODO: Support submodules besides query
        return vec![action];
    }
    // action=query
    let mut modules = vec![];
    for param in ["prop", "list", "meta"] {
        if let Some(value) = params.params.get(param) {
            let split = value.split('|');
            for sp in split {
                modules.push(format!("query+{}", sp));
            }
        }
    }

    modules
}

// FIXME merge with `make_struct`
// TODO: Consider turning this into a ToTokens impl?
fn make_struct_fields(
    ident: &Ident,
    fields: &[&metadata::Field],
) -> TokenStream2 {
    let stream = quote! {
        #[doc(hidden)]
        #[derive(Debug, Clone, ::mwapi_responses::serde::Deserialize)]
        #[serde(crate = "::mwapi_responses::serde")]
        pub struct #ident {
            #(#fields)*
        }
    };

    stream
}

// FIXME merge with `make_struct_fields`
// TODO: Consider turning this into a ToTokens impl?
fn make_struct(
    ident: &Ident,
    fields: &HashMap<String, TokenStream2>,
) -> TokenStream2 {
    let mut names = vec![];
    let mut types = vec![];
    for (name, ident) in fields.iter() {
        names.push(format_ident!("{}", name));
        types.push(ident);
    }

    let stream = quote! {
        #[doc(hidden)]
        #[derive(Debug, Clone, ::mwapi_responses::serde::Deserialize)]
        #[serde(crate = "::mwapi_responses::serde")]
        pub struct #ident {
            #(pub #names: #types,)*
        }
    };

    stream
}

/// Turn the HashMap of params into a slice that returns
/// `&[(&str, &str)]`, which is the format used by mwapi and trivially
/// convertable into a HashMap<String, String> for mediawiki-rs.
impl ToTokens for Params {
    fn to_tokens(&self, tokens: &mut TokenStream2) {
        let keys = self.params.keys();
        let values = self.params.values();

        let stream = quote! {
            &[
                #((#keys, #values),)*
            ]
        };
        stream.to_tokens(tokens);
    }
}

fn build_modules(
    prefix: &Ident,
    sub_ident: &Ident,
    modules: &[String],
    params: &Params,
) -> TokenStream2 {
    #[cfg(feature = "dbg")]
    dbg!(modules);
    let mut structs = vec![];
    let mut sub = HashMap::new();
    for name in modules {
        let info = metadata::Metadata::new(name);
        let props: Vec<String> = match params.params.get(&info.prop) {
            Some(value) => {
                value.split('|').map(|prop| prop.to_string()).collect()
            }
            None => {
                // FIXME: implement defaults
                unimplemented!("props must be explicitly specified right now");
            }
        };
        let fields: Vec<&metadata::Field> =
            props.iter().map(|prop| info.get_field(prop)).collect();
        // TODO: Would be nice to make this TitleCase
        let name_for_rust = name.replace('+', "");
        let module_ident = format_ident!("{}{}", prefix, name_for_rust);
        #[cfg(feature = "dbg")]
        dbg!(&fields);
        structs.push(make_struct_fields(&module_ident, &fields));
        match info.mode.as_ref() {
            "list" => {
                sub.insert(info.fieldname, quote! { Vec<#module_ident> });
            }
            _ => unimplemented!("Support for {} modules", info.mode),
        }
    }

    let sub_struct = make_struct(sub_ident, &sub);

    // Merge it all together
    let stream = quote! {
        #sub_struct

        #(#structs)*
    };

    stream
}

fn impl_query(ast: &DeriveInput, params: &Params) -> TokenStream2 {
    //    dbg!(ast);
    let prefix = &ast.ident;
    let action = format_ident!("{}", params.action());
    let sub_ident = format_ident!("{}SubBody", prefix);
    let modules = get_modules(params);
    let structs = build_modules(prefix, &sub_ident, &modules, params);

    quote! {
        #[doc = "Autogenerated MediaWiki API response body"]
        #[derive(Debug, Clone, ::mwapi_responses::serde::Deserialize)]
        #[serde(crate = "::mwapi_responses::serde")]
        pub struct #prefix {
            #[serde(default)]
            batchcomplete: bool,
            #[serde(rename = "continue")]
            #[serde(default)]
            continue_: ::std::collections::HashMap<String, String>,
            // TODO: is the top-level key always the same name as the action?
            // Hopefully.
            #action: #sub_ident,
        }

        impl #prefix {
            pub fn params() -> &'static [(&'static str, &'static str)] {
                #params
            }

            // This is mostly a helper for mediawiki-rs, but shouldn't
            // be necessary long-term
            pub fn from_value(val: ::mwapi_responses::serde_json::Value) -> ::mwapi_responses::serde_json::Result<Self> {
                ::mwapi_responses::serde_json::from_value(val)
            }
        }

        #structs
    }
}
