/*
Copyright (C) 2020-2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
//! mwapi_responses
//! ===============
//!
//! The `mwapi_responses` crate provides strict typing for dynamic MediaWiki
//! Action API queries. The goal is to faithfully represent what the API's
//! JSON structure is while saving people from manually writing out
//! serde-compatible structs for every query.
//!
//! A list module:
//! ```
//! use mwapi_responses::prelude::*;
//! #[query(
//!     list = "logevents",
//!     leaction = "newusers/create",
//!     leprop = "user|type",
//!     lelimit = "100"
//! )]
//! struct Response;
//! ```
//!
//! This creates a `Response` struct that matches the API response for the given
//! query. Roughly, it will expand to:
//!
//! ```ignore
//! #[derive(Debug, Clone, serde::Deserialize)]
//! pub struct Response {
//!     #[serde(default)]
//!     pub batchcomplete: bool,
//!     #[serde(rename = "continue")]
//!     #[serde(default)]
//!     pub continue_: HashMap<String, String>,
//!     pub query: ResponseBody,
//! }
//!
//! #[derive(Debug, Clone, serde::Deserialize)]
//! pub struct ResponseBody {
//!     pub logevents: Vec<ResponseItemlogevents>,
//! }
//!
//! #[derive(Debug, Clone, serde::Deserialize)]
//! pub struct ResponseItemlogevents {
//!     pub user: String,
//!     #[serde(rename = "type")]
//!     pub type_: String,
//! }
//! ```
//!
//! Or getting various properties from pages:
//! ```
//! use mwapi_responses::prelude::*;
//! #[query(
//!     prop="info|revisions",
//!     inprop="url",
//!     rvprop="ids"
//! )]
//! struct Response;
//! ```
//! which expands to:
//! ```ignore
//! #[derive(Debug, Clone, serde::Deserialize)]
//! pub struct Response {
//!     #[serde(default)]
//!     pub batchcomplete: bool,
//!     #[serde(rename = "continue")]
//!     #[serde(default)]
//!     pub continue_: HashMap<String, String>,
//!     pub query: ResponseBody,
//! }
//!
//! #[derive(Debug, Clone, serde::Deserialize)]
//! pub struct ResponseBody {
//!     pub pages: Vec<ResponseItem>,
//! }
//!
//! #[derive(Debug, Clone, serde::Deserialize)]
//! pub struct ResponseItem {
//!     pub canonicalurl: String,
//!     pub contentmodel: String,
//!     pub editurl: String,
//!     pub fullurl: String,
//!     pub lastrevid: Option<u32>,
//!     pub length: Option<u32>,
//!     #[serde(default)]
//!     pub missing: bool,
//!     #[serde(default)]
//!     pub new: bool,
//!     pub ns: i32,
//!     pub pageid: Option<u32>,
//!     pub pagelanguage: String,
//!     pub pagelanguagedir: String,
//!     pub pagelanguagehtmlcode: String,
//!     #[serde(default)]
//!     pub redirect: bool,
//!     pub title: String,
//!     pub touched: Option<String>,
//!     #[serde(default)]
//!     pub revisions: Vec<ResponseItemrevisions>,
//! }
//!
//! #[derive(Debug, Clone, serde::Deserialize)]
//! pub struct ResponseItemrevisions {
//!     pub parentid: u32,
//!     pub revid: u32,
//! }
//! ```
//!
//! In both cases, you can easily iterate over `ResponseItemlogevents`
//! and `ResponseItem` respectively by calling `Response::items()`.
//!
//! ## Field naming
//! Fields are renamed if they can't be used as fields in Rust, like `continue`
//! or `type`. In these cases, an underscore is appended.
//!
//! ## Supported modules
//! The metadata that powers this crate is manually gathered. To see if
//! a specific module is supported, look at the [Git repository](https://gitlab.com/mwbot-rs/mwbot/-/tree/master/mwapi_responses_derive/data).
//! Contributions for missing modules or parameters are always welcome.
//!
//! ## Library agnostic
//! This crate does not implement or support any one specific API library,
//! rather it aims to just provide types and helpers to enable you to run
//! your API requests however you'd like.
//!
//! ## Helper functions
//! To avoid repeating the parameters in multiple places, you can call
//! `Response::params()` to get a `&[(&str, &str)]` of the parameters that
//! were provided to the `#[query(...)]` macro.
//!
//! ## Not yet implemented
//! There is no special support for continuing queries. In the future some
//! merge()-like function might be provided.
pub use mwapi_responses_derive::query;
pub use serde;
pub mod prelude {
    pub use crate::{query, ApiResponse};
}
use std::collections::HashMap;

pub mod protection;
pub mod redirects;

pub trait ApiResponse<T> {
    /// Get the request params to send to the API
    fn params() -> &'static [(&'static str, &'static str)];

    /// Iterate over the main items in this request
    fn items(&self) -> &[T];

    fn get_continue(&self) -> &HashMap<String, String>;

    fn has_continue(&self) -> bool {
        !self.get_continue().is_empty()
    }
}
