/*
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

use serde::Deserialize;
use std::fmt::{Display, Formatter};
use thiserror::Error as ThisError;

#[derive(Clone, Copy, Debug)]
pub enum ErrorFormat {
    Html,
    Wikitext,
    Plaintext,
}

impl Display for ErrorFormat {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        let val = match &self {
            ErrorFormat::Html => "html",
            ErrorFormat::Wikitext => "wikitext",
            ErrorFormat::Plaintext => "plaintext",
        };
        write!(f, "{}", val)
    }
}

impl Default for ErrorFormat {
    fn default() -> Self {
        Self::Plaintext
    }
}

#[derive(Clone, Debug, Deserialize)]
pub struct ApiError {
    pub code: String,
    pub text: String,
    pub module: Option<String>,
}

impl Display for ApiError {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "API error: (code: {}): {}", self.code, self.text)
    }
}

#[derive(ThisError, Debug)]
pub enum Error {
    /// A HTTP error like a 4XX or 5XX status code
    #[error("HTTP error")]
    HttpError(#[from] reqwest::Error),
    /// Invalid header value, likely if the provided OAuth2 token
    /// was invalid
    #[error("Invalid header value")]
    InvalidHeaderValue(#[from] reqwest::header::InvalidHeaderValue),
    /// Error when decoding the JSON response from the API
    #[error("JSON error")]
    JsonError(#[from] serde_json::Error),
    /// Error if unable to get request concurrency lock
    #[error("Unable to get request lock")]
    LockError(#[from] tokio::sync::AcquireError),
    /// Unable to fetch a CSRF token
    #[error("Unable to get token `{0}`")]
    TokenError(String),
    /// Any arbitrary error returned by the MediaWiki API
    #[error("{0}")]
    ApiError(ApiError),
    /// An error where we don't know what to do nor have
    /// information to report back
    #[error("Unknown error: {0}")]
    UnknownError(String),
}
