/*
Copyright (C) 2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

use crate::error::ApiError;
use crate::responses;
use crate::{tokens::TokenStore, Error, ErrorFormat, Result};
use log::debug;
use reqwest::{header, Client as HttpClient, Request, Response};
use serde_json::Value;
use std::collections::HashMap;
use std::sync::Arc;
use tokio::sync::{RwLock, Semaphore};

/// Build a new API client.
/// ```
/// # use mwapi::{Client, Result};
/// # async fn doc() -> Result<()> {
/// let client: Client = Client::builder("https://example.org/w/api.php")
///     .set_oauth2_token("foobar")
///     .set_errorformat(mwapi::ErrorFormat::Html)
///     .build().await?;
/// # Ok(())
/// # }
/// ```
#[derive(Clone, Debug, Default)]
pub struct Builder {
    api_url: String,
    concurrency: usize,
    maxlag: Option<u32>,
    user_agent: Option<String>,
    oauth2_token: Option<String>,
    errorformat: ErrorFormat,
    botpassword: Option<BotPassword>,
}

#[derive(Clone, Debug)]
struct BotPassword {
    username: String,
    password: String,
}

impl Builder {
    /// Create a new `Builder` instance. Typically you will use
    /// `Client::builder()` instead.
    pub fn new(api_url: &str) -> Self {
        Self {
            api_url: api_url.to_string(),
            concurrency: 1,
            ..Default::default()
        }
    }

    /// Create a `Builder` with defaults aimed for bot operation.
    /// Typically you will use `Client::bot_builder()` instead.
    pub fn new_bot(api_url: &str) -> Self {
        Self::new(api_url).set_maxlag(5)
    }

    /// Create a `Builder` with defaults aimed for interactive operation.
    /// Typically you will use `Client::interactive_builder()` instead.
    pub fn new_interactive(api_url: &str) -> Self {
        Self::new(api_url)
    }

    /// Actually build the `Client` instance.
    pub async fn build(self) -> Result<Client> {
        let client = Client {
            api_url: self.api_url,
            http: HttpClient::builder()
                .cookie_store(true)
                .user_agent(
                    self.user_agent
                        .unwrap_or(format!("mwapi-rs/{}", crate::VERSION)),
                )
                .build()?,
            tokens: Arc::new(RwLock::new(TokenStore::default())),
            semaphore: Arc::new(Semaphore::new(self.concurrency)),
            oauth2_token: self.oauth2_token,
            errorformat: self.errorformat,
            maxlag: self.maxlag,
        };
        if let Some(botpassword) = self.botpassword {
            client.login(&botpassword).await?;
        }
        Ok(client)
    }

    /// Set a custom User-agent. Ideally follow the [Wikimedia User-agent policy](https://meta.wikimedia.org/wiki/User-Agent_policy).
    pub fn set_user_agent(mut self, user_agent: &str) -> Self {
        self.user_agent = Some(user_agent.to_string());
        self
    }

    /// Set an [OAuth2 token](https://www.mediawiki.org/wiki/OAuth/For_Developers#OAuth_2)
    /// for authentication
    pub fn set_oauth2_token(mut self, oauth2_token: &str) -> Self {
        self.oauth2_token = Some(oauth2_token.to_string());
        self
    }

    /// Set the format error messages from the API should be in
    pub fn set_errorformat(mut self, errorformat: ErrorFormat) -> Self {
        self.errorformat = errorformat;
        self
    }

    /// Set how many requests should be processed in parallel. On Wikimedia
    /// wikis, you shouldn't exceed the default of 1 without getting permission
    /// from a sysadmin.
    pub fn set_concurrency(mut self, concurrency: usize) -> Self {
        self.concurrency = concurrency;
        self
    }

    /// Pause when the servers are lagged for how many seconds?
    /// Typically bots should set this to 5, while interactive
    /// usage should be much higher.
    ///
    /// See [mediawiki.org](https://www.mediawiki.org/wiki/Special:MyLanguage/Manual:Maxlag_parameter)
    /// for more details.
    pub fn set_maxlag(mut self, maxlag: u32) -> Self {
        self.maxlag = Some(maxlag);
        self
    }

    pub fn set_botpassword(mut self, username: &str, password: &str) -> Self {
        self.botpassword = Some(BotPassword {
            username: username.to_string(),
            password: password.to_string(),
        });
        self
    }
}

#[derive(Clone, Debug)]
pub struct Client {
    api_url: String,
    http: HttpClient,
    tokens: Arc<RwLock<TokenStore>>,
    semaphore: Arc<Semaphore>,
    oauth2_token: Option<String>,
    errorformat: ErrorFormat,
    maxlag: Option<u32>,
}

impl Client {
    /// Get a `Builder` instance to further customize the API `Client`.
    /// The API URL should be the absolute path to [api.php](https://www.mediawiki.org/wiki/API:Main_page).
    pub fn builder(api_url: &str) -> Builder {
        Builder::new(api_url)
    }

    /// Same as `Client::builder`, but with defaults tuned for bot operation.
    pub fn bot_builder(api_url: &str) -> Builder {
        Builder::new_bot(api_url)
    }

    /// Same as `Client::builder`, but with defaults tuned for interactive operation.
    pub fn interactive_builder(api_url: &str) -> Builder {
        Builder::new_interactive(api_url)
    }

    /// Get an API `Client` instance. The API URL should be the absolute
    /// path to [api.php](https://www.mediawiki.org/wiki/API:Main_page).
    pub async fn new(api_url: &str) -> Result<Self> {
        Builder::new(api_url).build().await
    }

    /// Get headers that should be applied to every request
    fn headers(&self) -> Result<header::HeaderMap> {
        let mut headers = header::HeaderMap::new();
        if let Some(token) = &self.oauth2_token {
            headers.insert(
                header::AUTHORIZATION,
                format!("Bearer {}", token).parse()?,
            );
        }

        Ok(headers)
    }

    async fn login(&self, botpassword: &BotPassword) -> Result<()> {
        // Don't use a cached token, we need a fresh one
        let token = self.tokens.write().await.load("login", &self).await?;
        let resp = self
            .post(&[
                ("action", "login"),
                ("lgname", &botpassword.username),
                ("lgpassword", &botpassword.password),
                ("lgtoken", &token),
            ])
            .await?;
        let login_resp: responses::LoginResponse =
            serde_json::from_value(resp)?;
        // Convert "result": "Failed" into API errors
        if login_resp.login.result == "Failed" {
            Err(match login_resp.login.reason {
                Some(reason) => Error::ApiError(reason),
                None => Error::UnknownError("Login failed".to_string()),
            })
        } else {
            Ok(())
        }
    }

    /// Make an arbitrary API request using HTTP GET.
    pub async fn get<P: AsRef<str>>(&self, params: &[(P, P)]) -> Result<Value> {
        let mut params: HashMap<String, String> = params
            .iter()
            .map(|(key, val)| {
                (key.as_ref().to_string(), val.as_ref().to_string())
            })
            .collect();
        params.insert("format".to_string(), "json".to_string());
        params.insert("formatversion".to_string(), "2".to_string());
        params.insert("errorformat".to_string(), self.errorformat.to_string());
        if let Some(maxlag) = self.maxlag {
            params.insert("maxlag".to_string(), maxlag.to_string());
        }
        let req = self
            .http
            .get(&self.api_url)
            .headers(self.headers()?)
            .query(&params)
            .build()?;
        let _lock = self.semaphore.acquire().await?;
        log_request(&req);
        let resp = self.http.execute(req).await?;
        log_response(&resp);
        drop(_lock);
        let value: Value = resp.error_for_status()?.json().await?;
        match value.get("errors") {
            Some(errors) => {
                let errors: Vec<ApiError> =
                    serde_json::from_value(errors.clone())?;
                // FIXME: What about the other errors?
                Err(Error::ApiError(errors[0].clone()))
            }
            None => Ok(value),
        }
    }

    /// Make an API POST request with a [CSRF token](https://www.mediawiki.org/wiki/API:Tokens).
    /// The correct token will automatically be fetched, and in case of a
    /// bad token error (if it expired), a new one will automatically be
    /// fetched.
    pub async fn post_with_token<P: AsRef<str>>(
        &self,
        token: &str,
        params: &[(P, P)],
    ) -> Result<Value> {
        let mut params: HashMap<_, _> = params
            .iter()
            .map(|(key, val)| {
                (key.as_ref().to_string(), val.as_ref().to_string())
            })
            .collect();
        let token = match self.tokens.read().await.get(token) {
            Some(token) => token,
            None => self.tokens.write().await.load(token, &self).await?,
        };
        params.insert("token".to_string(), token.to_string());
        let params: Vec<_> = params.iter().collect();
        self.post(params.as_slice()).await
    }

    /// Make an API POST request
    pub async fn post<P: AsRef<str>>(
        &self,
        params: &[(P, P)],
    ) -> Result<Value> {
        let mut params: HashMap<String, String> = params
            .iter()
            .map(|(key, val)| {
                (key.as_ref().to_string(), val.as_ref().to_string())
            })
            .collect();
        params.insert("format".to_string(), "json".to_string());
        params.insert("formatversion".to_string(), "2".to_string());
        params.insert("errorformat".to_string(), self.errorformat.to_string());
        if let Some(maxlag) = self.maxlag {
            params.insert("maxlag".to_string(), maxlag.to_string());
        }
        let req = self
            .http
            .post(&self.api_url)
            .headers(self.headers()?)
            .form(&params)
            .build()?;
        let _lock = self.semaphore.acquire().await?;
        log_request(&req);
        let resp = self.http.execute(req).await?;
        log_response(&resp);
        drop(_lock);
        let value: Value = resp.error_for_status()?.json().await?;
        match value.get("errors") {
            Some(errors) => {
                let errors: Vec<ApiError> =
                    serde_json::from_value(errors.clone())?;
                Err(Error::ApiError(errors[0].clone()))
            }
            None => Ok(value),
        }
    }
}

fn log_request(req: &Request) {
    let method = req.method().to_string();
    let url = req.url().to_string();
    // TODO: form body?
    debug!("Sending: HTTP {}: {}", method, url);
}

fn log_response(resp: &Response) {
    let status = resp.status().as_u16();
    let request_id = match resp.headers().get("x-request-id") {
        // Not worth logging an error if the header is invalid utf-8
        Some(val) => val.to_str().unwrap_or("unknown"),
        None => "unknown",
    };
    let url = resp.url().to_string();
    debug!("Received: {} (req: {}): {}", status, request_id, url);
}

#[cfg(test)]
mod tests {
    use super::*;

    #[tokio::test]
    async fn test_basic_get() {
        let client = Client::new("https://www.mediawiki.org/w/api.php")
            .await
            .unwrap();
        let resp = client
            .get(&[("action", "query"), ("meta", "siteinfo")])
            .await
            .unwrap();
        assert_eq!(
            resp["query"]["general"]["sitename"].as_str().unwrap(),
            "MediaWiki"
        );
    }

    #[tokio::test]
    async fn test_basic_errors() {
        let client = Client::new("https://www.mediawiki.org/w/api.php")
            .await
            .unwrap();
        let error = client.get(&[("action", "nonexistent")]).await.unwrap_err();
        assert_eq!(
            &error.to_string(),
            "API error: (code: badvalue): Unrecognized value for parameter \"action\": nonexistent."
        );
    }

    #[tokio::test]
    async fn test_builder() {
        let client = Client::builder("https://www.mediawiki.org/w/api.php")
            .set_oauth2_token("foobarbaz")
            .build()
            .await
            .unwrap();
        assert_eq!(client.oauth2_token, Some("foobarbaz".to_string()));
    }

    #[tokio::test]
    async fn test_login() {
        let username = std::env::var("MWAPI_USERNAME");
        let password = std::env::var("MWAPI_PASSWORD");
        if username.is_err() || password.is_err() {
            // Skip
            return;
        }
        let username = username.unwrap();
        let password = password.unwrap();
        let client = Client::builder("https://test.wikipedia.org/w/api.php")
            .set_botpassword(&username, &password)
            .build()
            .await
            .unwrap();
        let resp = client
            .get(&[("action", "query"), ("meta", "userinfo")])
            .await
            .unwrap();
        dbg!(&resp);
        // Check the botpassword username ("Foo@something") starts with the real wiki username ("Foo")
        assert!(&username
            .starts_with(resp["query"]["userinfo"]["name"].as_str().unwrap()));
    }

    #[tokio::test]
    async fn test_bad_login() {
        let error = Client::builder("https://test.wikipedia.org/w/api.php")
            .set_botpassword("ThisAccountDoesNotExistPlease", "password")
            .build()
            .await
            .unwrap_err();
        match error {
            Error::ApiError(err) => {
                assert_eq!(&err.code, "wrongpassword");
            }
            err => {
                panic!("Unexpected error: {:?}", err);
            }
        }
    }
}
