use anyhow::{bail, format_err, Result};
use jacklog::{debug, error, info};
use std::convert::TryFrom;
use std::os::unix::prelude::CommandExt;
use std::path::PathBuf;
use std::process::Command;
use std::{env, fs};
use structopt::StructOpt;

/// `RealPath` represents a path that is known to actually exist on the
/// filesystem, and is fully qualified.
///
/// # Errors
///
/// Will return an error if the path does not exist on the filesystem.
struct RealPath(PathBuf);

impl TryFrom<PathBuf> for RealPath {
    type Error = anyhow::Error;

    fn try_from(p: PathBuf) -> Result<Self> {
        if !p.exists() {
            bail!("path {:?} does seem to exist on the filesystem", p);
        }

        Ok(Self(p))
    }
}

/// Create or join a tmux session based on the name of the current directory.
#[derive(Debug, StructOpt)]
#[structopt(rename_all = "kebab-case")]
#[structopt(setting = structopt::clap::AppSettings::ColoredHelp)]
struct Cli {
    /// Directory to use as the root of a new tmux session. Pass the flag
    /// multiple times to increase verbosity, up to a maximum of 3.
    #[structopt(short, long, parse(from_occurrences))]
    verbose: usize,

    /// Directory to use as the root of a new tmux session. Default behavior is
    /// to use the path as the session name as well.
    path: Option<PathBuf>,
}

fn main() -> Result<()> {
    // Parse the options. We need to do this before we set up logging, so we
    // can figure out how verbose to be.
    let opts = Cli::from_args();

    // Set up logging, based on how many -v flags were passed.
    jacklog::init(Some(match opts.verbose {
        0 => &"warn",
        1 => &"info",
        2 => &"debug",
        _ => &"trace",
    }))?;
    debug!("{:?}", &opts);

    // Check to make sure we're not in an existing session.
    if env::var("TMUX").is_ok() {
        bail!("muxme can't be used from within existing tmux sessions");
    }

    // Convert a relative path to an absolute one if necesary.
    let path = if let Some(p) = opts.path {
        if p.is_absolute() {
            debug!(path = ?p, "found absolute path");
            p
        } else {
            let abs = env::current_dir()?.join(p);
            debug!(path = ?abs, "converting relative path to absolute");

            abs
        }
    } else {
        let p = env::current_dir()?;
        debug!(path = ?p, "no path specified; using current working directory");

        p
    };

    // Make sure the path exists.
    let path = RealPath::try_from(path)?;
    debug!(path = ?path.0, "path appears to exist");

    // Check to see whether there's a .tmux.conf file in the CWD, and if so,
    // just pass it as the arg and don't worry about managing the session config.
    match fs::metadata(path.0.join(".tmux.conf")) {
        Ok(f) => {
            debug!("found a .tmux.conf dile in tmux root; will try to use it");

            if !f.is_file() {
                error!(".tmux.conf does not appear to be a file; cannot use");
            }

            // Spawn tmux, using the .tmux.conf file.
            error!(error = %Command::new("tmux")
                .args(&["-f", ".tmux.conf", "attach"])
                .exec());
        }
        Err(e) => debug!(msg = %e, "no .tmux.conf file in tmux root; skipping"),
    }

    // Figure out the name to use by expanding the path and getting the current
    // directory.
    // TODO: Check to make sure it's ascii.
    let name = path
        .0
        .file_name()
        .ok_or_else(|| {
            format_err!(
                "specified path does not appear to end in a reasonable name for a tmux session"
            )
        })?
        .to_str()
        .ok_or_else(|| format_err!("name does not appear to be valid unicode"))?;
    info!(?name, "extracting session name from path");

    // If we don't have an existing session, we need to create one.
    if !Command::new("tmux")
        .args(&["has-session", "-t", name])
        .status()?
        .success()
    {
        // Otherwise, we must need to create our own session, so we'll do that now.

        // Convert the path to a string so we can set in the tmux invocation.
        let path = path
            .0
            .to_str()
            .ok_or_else(|| format_err!("specified path does not appear to be valid unicode"))?;

        info!(?name, ?path, "launching new tmux session");
        if !Command::new("tmux")
            .args(&["new-session", "-s", name, "-d", "-c", path])
            .status()?
            .success()
        {
            bail!("encountered error launching new session");
        }
    }

    // Exec tmux, attaching to the existing session.
    error!(error = %Command::new("tmux")
                .args(&["attach", "-t", name])
                .exec());

    Ok(())
}
