// This file holds Integration tests for multiple codecs together.
use chain::Chain;
use codec;
use codec::registry::CodecRegistry;
use codec::tests;

fn check_with_size(name: &str, size: usize, inp: &[u8], outp: &[u8]) {
    let reg = CodecRegistry::new();
    let c = Chain::new(&reg, name, size, true);
    assert_eq!(c.transform(inp.to_vec()).unwrap(), outp);
    let c = Chain::new(&reg, name, size, true);
    assert_eq!(c.reverse().transform(outp.to_vec()).unwrap(), inp);
    let c = Chain::new(&reg, name, size, false);
    assert_eq!(c.reverse().transform(outp.to_vec()).unwrap(), inp);
}

fn check(name: &str, inp: &[u8], outp: &[u8]) {
    for i in vec![5, 6, 7, 8, 12, 512] {
        check_with_size(name, i, inp, outp);
    }
}

fn check_small_buffer(name: &str, size: usize) {
    let reg = CodecRegistry::new();
    let c = Chain::new(&reg, name, size, true);
    let err = c.transform(tests::BYTE_SEQ.to_vec()).unwrap_err();
    let err = err.get_ref();
    let err: Option<&codec::Error> = err.and_then(|e| e.downcast_ref());
    match err {
        Some(&codec::Error::SmallBuffer) => (),
        Some(e) => panic!("wrong error: {}", e),
        None => panic!("wrong error"),
    }
}

#[test]
fn too_small_buffer() {
    check_small_buffer("base64:wrap", 1);
    check_small_buffer("base64:wrap", 2);
    check_small_buffer("base64:wrap", 3);
}

#[test]
fn round_trip() {
    tests::round_trip("base64:wrap");
    tests::round_trip("base32:wrap");
    tests::round_trip("url64:wrap");
}

#[test]
fn known_breakages() {
    check("base64:wrap,length=76", b"\xeb\xfa\x8e\x7e\x43\xa4\x3b\xc9\xec\xce\xec\xf7\x43\xb7\x73\xc2\x3f\x5c\x6d\x02\xf0\x9e\xb0\x80\x24\x41\xd7\x73\x57\x13\x9a\xcd\xc0\x38\xc4\x56\xdb\x46\x13\x79\x60\x5f\xf3\x69\xb3\x29\xe3\xde\xfc\xea\xad\x5f\xda\xf9\x4b\xb6\xaa\x9c\x87\xfd\x3a\x77\xb2\xe7\xe5\xdf\x94\x50\x8d\x5e\xcb\x98\x07\x8c\x7e\x9f\x8b\x0d\x4a\xd5\xbf\x51\x2e\x8b\x9f\xed\x35\x55\x2a\xf1\x22\x58\x93\x69\x6a\xf9\xf8\x24\xaf\x4f\x41\xaa\xad\x30\x3e\x7a\x10\x5a\x68\xdb\xb0\xf6\x8e\x2d\x6c\x33\x86\x5e\x1c\xbb\x6b\xe0\x3b\x28\xb3\x51\x86\x93\x3e\x1d\xc5\x88\xbe\x04\xf2\xd6\x42\x37\x00\x42\x2a\x99\x6d\xd2\xc0\x5b\x35\x88\xd2\x09\x01\x73\x81\x33\xf7\xd1\x0d\x92\x3d\xb3\xe1\x12\x85\xaa\x14\x85\x16\x29\x0e\xf2\xbe\xba\xc4\x03\x8d\xd7\xe7\xa7\x8c\x5b\x6d\x9f\xfd\x9e\x0d\xcf\x92\x7a\x9a\x0a\xfe\x85\x00\x7a\x97\xda\x59\xb1\x61\x21\xd9\xe8\x5b\xb5\x7a\x18\xce\x5e\x6e\xa2\xe6\xb4\x46\xa0\x4e\xce\x6f\x77\xe3\x73\xe7\x9a\x01\xce\x2f\xe4\x25\xa1\xf5\x81\xd7\x98\xfd\xad\x04\x4f\x77\x15\xe8\xbe\x60\x82\x09\xa9\xeb\xa8\xf1\xa7\x3c\xc3\x51\xf4\x4f\x63\x78\x40\x88\x40\xe9\xa7\x91\x94\x1b\x5a\xc1\x93\x71\xd1\xa8\xc1\x25\xb5\xe3\x9a\x03\xd2\xcf\xb4\x1f\x7c\x16\x4b\xf6\x72\xc2\x50\x2d\x59\xd4\xc8\x9f\x83\xd9\xbd\x96\x66\x0b\x64\xbc\x3a\x85\x5c\xb0\x01\xe8\xca\xfd\x34\x8b\x56\x26\x85\xb5\xe1\x2e\x54\x65\x87\x82\xbc\xe8\xa5\x31\x38\xd2\x6a\x24\xc3\xa3\x5c\x05\x3f\x19\xf2\x93\x3a\xd4\x84\xf9\x62\x3f\x8c\xb4\xa6\xce\x31\xd6\xf2\x5e\x8a\x74\x8f\x28\x10\x6d\xac\xb7\xc5\x11\x0b\x85\x57\x50\x8e\xe8\x3c\x61\x09\x64\x36\x6b\x87\xfb\xcf\x79\xa5\xa6\xfd\x0c\x37\x66\x67\xf0\xd2\x59\x67\x93\xb3\xe2\x84\xb0\x31\x5b\x3b\xc9\x39\x89\xc7\x57\xdc\x1e\x4c\x6d\x77\x25\x1c\x0c\x10\x68\x1e\xcf\x2c\x05\x56\xd5\x2e\x2c\xf2\xd8\xce\x99\x4b\xfd\x05\xde\x58\x68\xef\xa4\x8b\xa1\x01\xce\x4d\x0f\xdc\x66\x2d\x66\x8a\xcd\x2a\x8b\x52\xed\x52\x0d\xe9\x57\xb2\x1c\x95\x4e\x9f\x9e\xc6\xd9\x81\xd2\x03\x8a\xa0\x16\xe5\x10\xdb\xad\x78\x11\x1b\x1b\xf9\xb6\x1d\x7f\x54\xa9\x3e\xe2\xe5\xbf\x67\x5e\xdb\xce\x5d\x32\x23\x8e\x89\x56\x30\x51\xc5\x97\x64\x76\x3a\x40\x82\x4b\x2d\xdd\xd6\xef\x61\xf1\x02\xd8\x73\x7a\x0a",
                        b"6/qOfkOkO8nszuz3Q7dzwj9cbQLwnrCAJEHXc1cTms3AOMRW20YTeWBf82mzKePe/OqtX9r5S7aq
nIf9Oney5+XflFCNXsuYB4x+n4sNStW/US6Ln+01VSrxIliTaWr5+CSvT0GqrTA+ehBaaNuw9o4t
bDOGXhy7a+A7KLNRhpM+HcWIvgTy1kI3AEIqmW3SwFs1iNIJAXOBM/fRDZI9s+EShaoUhRYpDvK+
usQDjdfnp4xbbZ/9ng3PknqaCv6FAHqX2lmxYSHZ6Fu1ehjOXm6i5rRGoE7Ob3fjc+eaAc4v5CWh
9YHXmP2tBE93Fei+YIIJqeuo8ac8w1H0T2N4QIhA6aeRlBtawZNx0ajBJbXjmgPSz7QffBZL9nLC
UC1Z1Mifg9m9lmYLZLw6hVywAejK/TSLViaFteEuVGWHgrzopTE40mokw6NcBT8Z8pM61IT5Yj+M
tKbOMdbyXop0jygQbay3xRELhVdQjug8YQlkNmuH+895pab9DDdmZ/DSWWeTs+KEsDFbO8k5icdX
3B5MbXclHAwQaB7PLAVW1S4s8tjOmUv9Bd5YaO+ki6EBzk0P3GYtZorNKotS7VIN6VeyHJVOn57G
2YHSA4qgFuUQ2614ERsb+bYdf1SpPuLlv2de285dMiOOiVYwUcWXZHY6QIJLLd3W72HxAthzego=
");
    check("base64:wrap", b"\x32\xa4\xc8\xd4\xc4\xe6\x29\x9e\xf1\x78\x66\xaf\x74\x24\xa8\x14\x81\x5f\x47\x8d\x1a\xd1\xac\xb1\x25\x6a\xe0\x8e\x6e\xcb\x9f\x7d\x07\xd8\x0a\x63\xd6\x46\xda\x95\x1c\xed\x81\x8b\x2d\x01\xbc\x34\xf7\xe1\x05", b"MqTI1MTmKZ7xeGavdCSoFIFfR40a0ayxJWrgjm7Ln30H2Apj1kbalRztgYstAbw09+EF");
}
