use std::io::Read;
use std::{fs, io::Write};
use std::fs::File;
use std::process::Command;
use std::str;
use std::vec::Vec;

use clap::{Arg, command};
use indicatif::ParallelProgressIterator;
use log::info;
use rayon::iter::{IntoParallelRefIterator, ParallelIterator};
use serde_derive::Serialize;
use simplelog::{LevelFilter, WriteLogger};
use uuid::Uuid;
use zip::ZipWriter;
use zip::write::FileOptions;

#[derive(Debug, Serialize)]
struct BookConfig {
    file_names: Vec<String>,
}

impl BookConfig {
    fn new(file_names: Vec<String>) -> Self {
        Self {
            file_names
        }
    }
}

fn generate_musicxml(input_path: &str, export_path: &str) {
    let mut command= Command::new("musescore");
    command.arg(input_path);
    command.arg("--export-to");
    command.arg(export_path);
    let command_output = command
        .output()
        .expect("failed to run musescore");
    
    let mut command_string = command.get_program().to_string_lossy().to_string();
    for arg in command.get_args() {
        command_string.push_str(arg.to_str().unwrap());
    }
    info!(
        "stdout for command:\n{}\nis\n{}",
        command_string,
        str::from_utf8(&command_output.stdout).unwrap()
    );
}

fn main() {
    let log_file = File::create("music_book_builder.log").unwrap();
    WriteLogger::init(LevelFilter::max(), simplelog::Config::default(), log_file).unwrap();

    let args = command!()
        .propagate_version(true)
        .about("Tool to build and manipulate music book files")
        .subcommand_required(true)
        .subcommand(clap::Command::new("generate")
            .about("Generate new music book file")
            .arg(Arg::new("input_directory")
                .short('i')
                .long("input-directory")
                .takes_value(true)
                .required(true)
                .help("Folder containing files to generate book from"))
            .arg(Arg::new("output_file")
                .short('o')
                .long("output-file")
                .takes_value(true )
                .required(true)
                .help("File to store the output to")))
        .get_matches();

    match args.subcommand() {
        Some(("generate", subcommand_args)) => {
            let input_path = subcommand_args.value_of("input_directory").unwrap();
            let export_file = subcommand_args.value_of("output_file").unwrap();

            let mut files_to_export = Vec::new();
            for dir_entry in fs::read_dir(input_path).unwrap() {
                let file = dir_entry.unwrap();
                if let Some(extension) =  file.path().extension() {
                    if extension.to_str().unwrap() == "mscz" {
                        files_to_export.push(file.path());
                    }
                }
            }

            let tmp_dir_name = format!("tmp.{}", Uuid::new_v4());
            let tmp_dir_name_str = tmp_dir_name.as_str();
            fs::create_dir(tmp_dir_name_str).unwrap();
            
            files_to_export.par_iter().progress().for_each(|file_path| {
                generate_musicxml(
                   file_path.to_str().unwrap(),
                   &format!("{}/{}.musicxml",
                       tmp_dir_name_str,
                       file_path.file_stem().unwrap().to_str().unwrap()
                    )
                );
            });

            let output_file = File::create(format!("{}.mb", export_file)).unwrap();

            let mut zip_writer = ZipWriter::new(&output_file);
            zip_writer.add_directory("musicxml", FileOptions::default()).unwrap();
            for item in fs::read_dir(tmp_dir_name_str).unwrap() {
                let dir_entry = item.unwrap();
                zip_writer.start_file(
                    format!("musicxml/{}", dir_entry.file_name().to_str().unwrap()),
                    FileOptions::default()).unwrap();
                let mut musicxml_file = File::open(dir_entry.path()).unwrap();
                let mut file_contents = String::new();
                musicxml_file.read_to_string(&mut file_contents).unwrap();
                zip_writer.write_all(file_contents.as_bytes()).unwrap();
            }

            let mut book_config_file_names = files_to_export.into_iter().map(|file| file.file_stem().unwrap().to_string_lossy().to_string()).collect::<Vec<String>>();
            book_config_file_names.sort();
            let book_config = BookConfig::new(book_config_file_names);
            let book_config_json_string = serde_json::to_string(&book_config).unwrap();
            zip_writer.start_file("book.json", FileOptions::default()).unwrap();
            zip_writer.write_all(book_config_json_string.as_bytes()).unwrap();
            zip_writer.finish().unwrap();
            /*let book_config_file_path= format!("{}/book.json", tmp_dir_name_str);
            let book_config_file_path_str= book_config_file_path.as_str();
            let mut book_config_file = File::create(book_config_file_path_str).unwrap();
            book_config_file.write(book_config_json_string.as_bytes()).unwrap();
            book_config_file.flush().unwrap();
            drop(book_config_file);

            tar_file.append_file("book.json", &mut File::open(book_config_file_path_str).unwrap()).unwrap();*/
        },
        _ => unreachable!("subcommand")
    }
    
}