use cursive::Cursive;
use cursive::theme::{Theme,PaletteColor,Color,BaseColor,BorderStyle};
use cursive::views::{LinearLayout,SelectView,ResizedView,TextView};
use cursive::utils::markup::StyledString;
use cursive::align::HAlign;
use cursive::traits::{Nameable,Scrollable};
use cursive::CursiveExt;
use cursive::event;

use crate::{library, player, views};

pub fn show_ui() {
    let player = player::Player::load_empty();
    let mut root = Cursive::default();
    let queue_view: SelectView<library::Song> =  SelectView::new();

    //initialize the library view
    let library = get_lib_view();

    // application theme
    let mut theme_def = Theme::default();
    theme_def.shadow = false;
    theme_def.palette[PaletteColor::Background] = Color::TerminalDefault;
    theme_def.palette[PaletteColor::Primary] = Color::Light(BaseColor::White);
    theme_def.palette[PaletteColor::View] = Color::TerminalDefault;
    theme_def.palette[PaletteColor::Highlight] = Color::from_256colors(250);
    theme_def.palette[PaletteColor::HighlightText] = Color::from_256colors(233);
    theme_def.palette[PaletteColor::HighlightInactive] = Color::from_256colors(242);
    theme_def.palette[PaletteColor::Secondary] = Color::Dark(BaseColor::Blue);
    theme_def.palette[PaletteColor::TitlePrimary] = Color::Light(BaseColor::Blue);
    theme_def.borders = BorderStyle::Simple;

    root.set_theme(theme_def);
    root.set_user_data(player);
    root.add_fullscreen_layer(
        LinearLayout::vertical()
        .child(    
            LinearLayout::horizontal()
                // music library
                .child(library)
                
                // play queue
                .child(
                    ResizedView::with_full_screen(
                        LinearLayout::vertical()
                            .child(TextView::new("Play Queue").h_align(HAlign::Center))
                            .child(views::VDivider::new().with_name("play_queue_width"))
                            .child(queue_view
                                .on_submit(|view,song|{
                                    view.with_user_data(|player: &mut player::Player| {
                                        player.set_pos(song.clone());
                                    });                    
                                })
                                .with_name("play_queue")
                                .scrollable()
                            )
                    )
                )
                .with_name("player")
        )
        .child(views::VDivider::new())
        .child(
            LinearLayout::horizontal()
            .child(ResizedView::with_full_width(TextView::empty().h_align(HAlign::Left).with_name("cur_song")))
            .child(TextView::empty().h_align(HAlign::Right).with_name("status"))
        )
        .child(views::PlaybackProgress::new().with_name("progress"))
    );

    // refresh event
    root.add_global_callback(event::Event::Refresh,|s|{
        let library = get_lib_view();
        let player = s.user_data::<player::Player>().unwrap();
        let song = player.song.clone();
        let queue = player.queue.clone();
        let now_playing = player.song.clone();
        let cur_time = player.cur_time();
        let secs = player.cur_time_secs();
        let repeat = player.get_repeat();
        let time = song.get_length_str();
        let state = player.get_state();
        let stopped = player.is_stopped();

        // keep the player's timing in sync with the playing audio
        player.update();

        // keep the progress bar in sync with the playing audio
        let mut total_width = 0;
        s.call_on_name("progress" , |prog: &mut views::PlaybackProgress|{
            total_width = prog.width;
            prog.update(secs, song.length)
        });

        // remove the library view if the terminal width is too small
        s.call_on_name("player", |view: &mut LinearLayout| {
            if total_width <= 120 && view.len() == 2 {
                view.remove_child(0);
            }
            
            if total_width > 120 && view.len() == 1 {
                view.insert_child(0, library);
            }
        });
        
        // the width of the play queue
        let mut width = 0;
        s.call_on_name("play_queue_width", |view: &mut views::VDivider| {
            width = view.width;
        });
        let mut subwidth = 0;
        if width != 0 {
            subwidth = (width - 7)/3;
        }

        s.call_on_name("cur_song", |view: &mut TextView|{
            let mut styled = StyledString::default();
            if stopped {
                view.set_content(state);
            }
            else {
                styled.append(StyledString::styled(&song.name,Color::Light(BaseColor::Blue)));
                styled.append(StyledString::plain("\n"));
                styled.append(StyledString::styled(&song.artist,Color::Light(BaseColor::White)));
                styled.append(StyledString::plain(" - "));
                styled.append(StyledString::styled(&song.album,Color::Light(BaseColor::Cyan)));
                styled.append(StyledString::plain(format!("\n{} / {} {}", cur_time, time, state)));
                view.set_content(styled);
            }
        });

        s.call_on_name("status" , |view: &mut TextView|{
            if stopped {
                view.set_content(format!("Repeat mode: {}",repeat));    
            }
            else {
                view.set_content(format!("\n\nRepeat mode: {}",repeat));
            }
        });

        s.call_on_name("play_queue", |view: &mut SelectView<library::Song>|{
            // prevent the selected item from being reset on update
            let selected = view.selected_id();
            
            view.clear();
            for song in queue.songs {
                let mut name = song.name.clone();
                let mut album_str = song.album.clone();
                let mut artist_str = song.artist.clone();
                while artist_str.len() < subwidth {
                    artist_str.push(' ');
                }
                while album_str.len() < subwidth {
                    album_str.push(' ');
                }
                while name.len() < subwidth {
                    name.push(' ');
                }
                let mut length = song.get_length_str();
                while length.len() < 7 {
                    length.push(' ');
                }

                let mut styled = StyledString::default();
                if song.name == now_playing.name {
                    styled.append(StyledString::styled(&artist_str[0..subwidth],Color::Light(BaseColor::White)));
                    styled.append(StyledString::styled(&name[0..subwidth],Color::Light(BaseColor::Blue)));
                    styled.append(StyledString::styled(&album_str[0..subwidth],Color::Light(BaseColor::Cyan)));
                    styled.append(StyledString::styled(&length[0..7],Color::Light(BaseColor::Magenta)));
                }
                else {
                    styled.append(StyledString::styled(&artist_str[0..subwidth],Color::Dark(BaseColor::White)));
                    styled.append(StyledString::styled(&name[0..subwidth],Color::Dark(BaseColor::Blue)));
                    styled.append(StyledString::styled(&album_str[0..subwidth],Color::Dark(BaseColor::Cyan)));
                    styled.append(StyledString::styled(&length[0..7],Color::Dark(BaseColor::Magenta)));
                }
                view.add_item(styled, song.clone());
            }

            // restore the user's chosen item (if there was one)
            if selected == None {
                return;
            }
            view.set_selection(selected.unwrap());
        });
    });

    // quit the application
    root.add_global_callback('q',|s|s.quit());

    // next song
    root.add_global_callback('n',|s|{
        s.with_user_data(|player: &mut player::Player| {
            player.next();
        });
    });

    // previous song
    root.add_global_callback('p',|s|{
        s.with_user_data(|player: &mut player::Player| {
            player.prev();
        });
    });

    // stop the player
    root.add_global_callback('s',|s|{
        s.with_user_data(|player: &mut player::Player| {
            player.stop();
        });
    });

    // pause/resume
    root.add_global_callback(' ',|s| {
        s.with_user_data(|player: &mut player::Player| {
            if player.is_stopped() {
                return;
            }
            if player.is_paused(){
                player.resume()
            }
            else {
                player.pause();
            }
        });
    });

    // shuffle the play queue
    root.add_global_callback('z',|s|{
        s.with_user_data(|player: &mut player::Player| {
            player.shuffle_queue();
        });
    });

    // set the repeat mode
    root.add_global_callback('r',|s|{
        s.with_user_data(|player: &mut player::Player| {
            player.set_repeat();
        });
    });

    //rewind the current song
    root.add_global_callback('R',|s|{
        s.with_user_data(|player: &mut player::Player| {
            if player.is_stopped() {
                return;
            }
            player.restart();
        });
    });
    root.set_fps(30);
    root.run();
}

fn get_lib_view() -> LinearLayout {
    let library = library::Library::load();
    let artist = library.artists[0].clone();
    let album = artist.albums[0].clone();
    let mut artist_selector: SelectView<library::Artist> = SelectView::new();
    let mut album_selector: SelectView<library::Album> = SelectView::new();
    let mut song_selector: SelectView<library::Song> = SelectView::new();

    for artist in library.artists {
        let name = format!("{:width$.30}",artist.name.clone(),width=30);
        artist_selector.add_item(name, artist);
    }

    for album in artist.albums {
        let name = format!("{:width$.30}",album.name.clone(),width=30);
        album_selector.add_item(name, album);
    }

    for song in album.songs {
        let name = format!("{:width$.30}",song.name.clone(),width=30);
        song_selector.add_item(name, song);
    }

    let view = LinearLayout::horizontal()
        // artist selector
        .child(
            ResizedView::with_fixed_width(30, 
                LinearLayout::vertical()
                .child(TextView::new("Artists").h_align(HAlign::Center))
                .child(views::VDivider::new())
                .child(artist_selector
                    .on_select(|s, artist|{
                        let album0 = artist.albums[0].clone();
                        fill_artist_albums(s, artist);
                        fill_album_songs(s, &album0);
                    })
                    .on_submit(|s,artist|{
                        s.with_user_data(|player: &mut player::Player| {
                            player.queue.add_artist(artist);
                            if player.is_empty() {
                                player.song = artist.albums[0].songs[0].clone();
                                player.play();
                            }
                        });                              
                    })
                    .with_name("artist_selector")
                    .scrollable()
                )
            )
        )
        .child(views::HDivider::new())

        // album selector
        .child(
            ResizedView::with_fixed_width(30, 
                LinearLayout::vertical()
                    .child(TextView::new("Albums").h_align(HAlign::Center))
                    .child(views::VDivider::new())
                    .child(album_selector
                        .on_select(fill_album_songs)
                        .on_submit(|s,album|{
                            s.with_user_data(|player: &mut player::Player| {
                                player.queue.add_album(album);
                                if player.is_empty() {
                                    player.song = album.songs[0].clone();
                                    player.play();
                                }
                            });
                        })
                        .with_name("album_selector")
                        .scrollable()
                    )
            )
        )
        .child(views::HDivider::new())

        // song selector
        .child(
            ResizedView::with_fixed_width(30, 
                LinearLayout::vertical()
                    .child(TextView::new("Songs").h_align(HAlign::Center))
                    .child(views::VDivider::new())
                    .child(song_selector
                        .on_submit(|s,song|{
                            s.with_user_data(|player: &mut player::Player| {
                                player.queue.add_song(song);
                                if player.is_empty() {
                                    player.song = song.clone();
                                    player.play();
                                }
                            });
                        })
                        .with_name("song_selector")
                        .scrollable()
                    )
            )
        )
        .child(views::HDivider::new());
    view
}

fn fill_artist_albums(view: &mut Cursive, artist: &library::Artist) {
    view.call_on_name("album_selector", |selector: &mut SelectView<library::Album>|{
        selector.clear();
        for album in &artist.albums {
            let name = format!("{:width$.30}",album.name.clone(),width=30);
            selector.add_item(name, album.clone());
        }
    });
}

fn fill_album_songs(view: &mut Cursive, album: &library::Album) {
    view.call_on_name("song_selector", |selector: &mut SelectView<library::Song>|{
        selector.clear();
        for song in &album.songs {
            let name = format!("{:width$.30}",song.name.clone(),width=30);
            selector.add_item(name, song.clone());
        }
    });
}