use std::time;
use lewton::inside_ogg::OggStreamReader;
use std::fs::File;
use std::env;
use std::thread;

use alto::{Alto, Mono, Stereo, Source, SourceState};
use crate::library;

#[derive(PartialEq)]
enum RepeatMode {
    One,
    All,
    Once
}

impl RepeatMode {
    pub fn desc(&self) -> String {
        match *self {
            RepeatMode::All => "All".to_string(),
            RepeatMode::One => "Current Song".to_string(),
            RepeatMode::Once => "No Repeat".to_string()
        }
    }
}

#[derive(PartialEq)]
enum PlayerState {
    Playing,
    Paused,
    Stopped
}

impl PlayerState {
    pub fn desc(&self) -> String {
        match *self {
            PlayerState::Paused => "[paused]".to_string(),
            PlayerState::Playing => "[playing]".to_string(),
            PlayerState::Stopped => "[stopped]".to_string()
        }
    }
}

pub struct Player {
    // the play queue (can be a playlist)
    pub queue: library::Playlist,

    //the currently playing song
    pub song: library::Song,

    // when did the song start (Unix epoch in seconds)?
    // 0 means the player is stopped
    pub begin: usize,

    // when should it end (Unix epoch in seconds)?
    // 0 means the player is stopped
    pub end: usize,

    // how much time is left of a paused song, 0 if it is playing
    pub time_left: usize,

    // the song the player is at in the playlist
    position: usize,

    // the state of the player
    state: PlayerState,

    // should one song, all songs, or no songs be repeated?
    repeat: RepeatMode,
}

impl Player {
    // load the player with a specific playlist
    pub fn load(list: &library::Playlist) -> Player {
        // the first song that will be played
        let first_song = list.songs[0].clone();

        // indicates a stopped player
        let begin = 0;
        let end = 0;

        Player{
            song: first_song,
            queue: list.clone(),
            begin,
            end,
            time_left:0,
            position: 0,
            state: PlayerState::Stopped,
            repeat: RepeatMode::Once
        }
    }

    // load the player with no playlist
    pub fn load_empty() -> Player {
        Player {
            song: library::Song::empty(),
            queue: library::Playlist::new("_"),
            begin: 0,
            end: 0,
            time_left:0,
            position: 0,
            state: PlayerState::Stopped,
            repeat: RepeatMode::Once
        }
    }

    // is the player empty?
    pub fn is_empty(&self) -> bool {
        self.song.name == "_"
    }

    // set the repeat mode
    pub fn set_repeat(&mut self) {
        let new_mode = match self.repeat {
            RepeatMode::All=>RepeatMode::Once,
            RepeatMode::Once=>RepeatMode::One,
            RepeatMode::One=>RepeatMode::All
        };

        self.repeat = new_mode;
    }

    // get the repeat mode
    pub fn get_repeat(&mut self) -> String{
        self.repeat.desc()
    }

    // get the player state
    pub fn get_state(&self) -> String {
        self.state.desc()
    }
    // play the current song
    pub fn play(&mut self) {
        self.state = PlayerState::Playing;
        let time = time::SystemTime::now().duration_since(time::SystemTime::UNIX_EPOCH).unwrap();
        self.song = self.queue.songs[self.position].clone();
        self.begin = time.as_secs() as usize;
        self.end = time.as_secs() as usize + self.song.length;
        self.time_left=0;
        play_vorbis(&self.song.path);
    }

    pub fn restart(&mut self) {
        env::set_var("ML_CMD", "STOP");
        self.play();
    }
    // next song
    pub fn next(&mut self) {
        if self.position == self.queue.songs.len()-1 {
            match self.repeat {
                RepeatMode::All => {
                    env::set_var("ML_CMD", "STOP");
                    self.position = 0;
                    self.song = self.queue.songs[0].clone();
                    self.play();
                    return;
                },
                RepeatMode::Once => {
                    return;
                },
                _=>{}
            }
        }
        env::set_var("ML_CMD", "STOP");
        if self.repeat != RepeatMode::One {
            self.position += 1;
        }
        self.song = self.queue.songs[self.position].clone();
        self.play();
    }

    // previous song
    pub fn prev(&mut self) {
        if self.position == 0 {
            self.position = self.queue.songs.len()-1;
        }
        else {
            self.position -= 1;
        }
        env::set_var("ML_CMD", "STOP");
        
        self.song = self.queue.songs[self.position].clone();
        self.play();
    }

    // resume paused song
    pub fn resume(&mut self) {
        let time = time::SystemTime::now().duration_since(time::SystemTime::UNIX_EPOCH).unwrap();
        env::set_var("ML_CMD", "_");
        self.end = self.time_left.clone() + time.as_secs() as usize;
        self.state = PlayerState::Playing;
    }

    // pause the song
    pub fn pause(&mut self) {
        env::set_var("ML_CMD", "PS");
        self.time_left = self.time_left();
        self.state = PlayerState::Paused;
    }

    // stop the player and clear the play queue
    pub fn stop(&mut self) {
        env::set_var("ML_CMD", "STOP");
        self.queue = library::Playlist::new("_");
        self.song = library::Song::empty();
        self.begin = 0;
        self.end = 0;
        self.position = 0;
        self.state = PlayerState::Stopped;
    }

    // shuffle the play queue
    pub fn shuffle_queue(&mut self) {
        if self.state == PlayerState::Stopped {
            return;
        }
        self.queue.shuffle(self.position);
        self.position = 0;
    }

    // get the current time of the song
    pub fn cur_time(&self) -> String {
        let mut seconds = self.song.length as u64 - self.time_left() as u64;
        if self.state == PlayerState::Paused {
            seconds = self.song.length as u64 - self.time_left as u64;
        }
        if self.state == PlayerState::Stopped {
            seconds = 0;
        }

        let sec_rem = seconds % 60;

        let minutes = (seconds - (seconds%60))/60;

        format!("{}:{:02}",minutes,sec_rem)
    }

    pub fn cur_time_secs(&self) -> usize {
        let mut seconds = self.song.length as u64 - self.time_left() as u64;
        if self.state == PlayerState::Paused {
            seconds = self.song.length as u64 - self.time_left as u64;
        }
        if self.state == PlayerState::Stopped {
            seconds = 0;
        }

        seconds as usize
    }

    // how much time is left of the song?
    pub fn time_left(&self) -> usize {
        if self.end == 0 || !self.is_playing(){
            return 0;
        }
        let time = time::SystemTime::now().duration_since(time::SystemTime::UNIX_EPOCH).unwrap();
        self.end - time.as_secs() as usize
    }

    // is the player paused
    pub fn is_paused(&self) -> bool {
        self.state == PlayerState::Paused
    }

    // is the player stopped
    pub fn is_stopped(&self) -> bool {
        self.state == PlayerState::Stopped
    }

    // is player playing a song?
    pub fn is_playing(&self) -> bool {
        self.state == PlayerState::Playing
    }

    // go to another song in the play queue
    pub fn set_pos(&mut self, song: library::Song) {
        env::set_var("ML_CMD", "STOP");
        // songs in the play queue
        let songs = self.queue.songs.clone();

        // the name of the to set
        let name = song.name.clone();

        // find the song's position in the play queue
        for (i,s) in songs.iter().enumerate() {
            if s.name == name {
                self.position = i;
            }
        }

        // set the song
        self.song = song;

        self.play()
    }

    // check if the player is still playing a song
    pub fn update(&mut self) {
        if self.state == PlayerState::Paused || self.state ==PlayerState::Stopped {
            return;
        }
        if self.time_left() == 1 {
            self.next();
        }
    }
    
}

// play an ogg vorbis file
// ogg vorbis is currently the only supported music format

fn play_vorbis<F: Into<String>>(file: F) {
    let file_str = file.into();
    let f = File::open(&file_str).expect(&format!("Where is this file again?\n{}",&file_str)[..]);
    let mut stream = OggStreamReader::new(f).unwrap();
    let al = Alto::load_default().expect("Could not load alto");//replace with cpal
    let device = al.open(None).expect("Could not open device");//replace with cpal
    let context = device.new_context(None).expect("Could not create context");//replace with cpal
    let mut source = context.new_streaming_source().expect("could not create streaming source!");//replace with cpal
    let sample_rate = stream.ident_hdr.audio_sample_rate as i32;

    if stream.ident_hdr.audio_channels > 2 {
        println!("Stream error: {} channels are too many!", stream.ident_hdr.audio_channels);
    }

    let builder = thread::Builder::new().name("audio".to_string());
    builder.spawn(move ||{
        let mut n = 0;
        while let Some(pck_samples) = stream.read_dec_packet_itl().unwrap() {
            let mut cmd = env::var("ML_CMD").unwrap();
            n += 1;
            let buf = match stream.ident_hdr.audio_channels {//replace with cpal
                1 => context.new_buffer::<Mono<i16>,_>(&pck_samples, sample_rate),//replace with cpal
                2 => context.new_buffer::<Stereo<i16>,_>(&pck_samples, sample_rate),//replace with cpal
                n => panic!("unsupported number of channels: {}", n),
            }.unwrap();

            source.queue_buffer(buf).unwrap();//replace with cpal
        
            if n == 25 {
                source.play();//replace with cpal
            }

            match cmd.as_str() {
                "STOP" => {
                    source.stop();//replace with cpal
                    env::set_var("ML_CMD", "_");
                    return;
                },
                "PS" =>{
                    source.pause();//replace with cpal

                    while cmd.as_str()=="PS" {
                        cmd = env::var("ML_CMD").unwrap();
                    }
                    source.play();//replace with cpal
                },
                _ => {}
            }
        }

        // loop until the playback ends
        while source.state() != SourceState::Stopped {//replace with cpal
            let mut cmd = env::var("ML_CMD").unwrap();
            match cmd.as_str() {
                "STOP" => {
                    source.stop();//replace with cpal

                    env::set_var("ML_CMD", "_");
                    return;
                },
                "PS" =>{
                    source.pause();//replace with cpal

                    while cmd.as_str()=="PS" {
                        cmd = env::var("ML_CMD").unwrap();
                    }
                    source.play();//replace with cpal
                },
                _ => {}
            }
        }
    }).expect("Unable to spawn audio thread!");
}