use std::env;
use rusqlite::{params, Connection};
use glob::glob;
use lewton::inside_ogg::OggStreamReader;
use std::fs::File;
use std::path::Path;
use std::fmt;
use ogg_metadata;
use ogg_metadata::AudioMetadata;
use rand::Rng;

//it is a pain in the ass to get strings from sqlite
struct SqlString {
    text: String
}

// song object
// contains information about a song
#[derive(Clone)]
pub struct Song {
    // what is the song's name?
    pub name: String,

    // what album is it from?
    pub album: String,

    // who created it?
    pub artist: String,

    // how long is it (in seconds)?
    pub length: usize,

    // file path
    pub path: String
}

impl Song {
    pub fn empty() -> Song {
        Song {
            name: "_".to_string(),
            album: "_".to_string(),
            artist: "_".to_string(),
            length: 0,
            path: "_".to_string()
        }
    }
    pub fn get_length_str(&self) -> String {
        let seconds = self.length;

        let sec_rem = seconds % 60;

        let minutes = (seconds - (seconds%60))/60;

        format!("{}:{:02}",minutes,sec_rem)
    }
}

impl fmt::Display for Song {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "Title: {}\nAlbum: {}\nArtist: {}\nLength in seconds: {}\nFile Path: {}",self.name,self.album,self.artist,self.get_length_str(),self.path)
    }
}

#[derive(Clone)]
pub struct Album {
    // what is the album's name?
    pub name: String,

    // who created it?
    pub artist: String,

    // what are the songs in it?
    pub songs: Vec<Song>

}

impl fmt::Display for Album {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        writeln!(f,"Title: {}",self.name).unwrap();
        writeln!(f,"Artist: {}",self.artist).unwrap();
        writeln!(f,"Songs:").unwrap();
        for song in &self.songs {
            writeln!(f,"  {}",song.name).unwrap();
        }
        write!(f," ")
    }
}
impl Album {
    pub fn init(name: &String, artist: &String) -> Self {
        Album {
            name: name.clone(),
            artist: artist.clone(),
            songs: Vec::new()
        }
    }
}

#[derive(Clone)]
pub struct Artist {
    // what is the artist's name?
    pub name: String,

    // what albums belong to them?
    pub albums: Vec<Album>
}

impl Artist {
    pub fn init(name: &String) -> Self {
        Artist {
            name: name.clone(),
            albums: Vec::new()
        }
    }
}

impl fmt::Display for Artist {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        writeln!(f,"Name: {}", self.name).unwrap();
        writeln!(f,"Albums:").unwrap();
        for album in &self.albums {
            writeln!(f, "  {}", album.name).unwrap();
        }
        write!(f," ")
    }
}

pub struct Library {
    pub artists: Vec<Artist>,

    pub num_songs: usize,
    pub num_albums: usize,
    pub num_artists: usize
}

impl fmt::Display for Library {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "Number of songs: {}\nNumber of albums: {}\nNumber of artists: {}", self.num_songs,self.num_albums,self.num_artists)
    }
}

impl Library {
    // loads in the library database or creates a new one if it does not exist
    pub fn load() -> Library {
        let mut songs: Vec<Song> = Vec::new();
        let mut album_names: Vec<String> = Vec::new();
        let mut albums: Vec<Album> = Vec::new();
        let mut artist_names: Vec<String> = Vec::new();
        let mut artists: Vec<Artist> = Vec::new();
        let [mut num_songs, mut num_albums, mut num_artists] = [0;3];
        let conn: rusqlite::Connection;
        if !Path::new(&format!("{}/.local/share/music-lounge/library.db",env::var("HOME").unwrap())).exists() {
            conn = gendb();
        }
        else {
            conn = Connection::open(format!("{}/.local/share/music-lounge/library.db",env::var("HOME").unwrap())).unwrap();
        }

        //get the list of songs
        let mut cmd = conn.prepare("SELECT * FROM library").unwrap();

        let song_iter = cmd.query_map([], |row| {
            Ok(
                Song {
                    name: row.get(0)?,
                    album: row.get(1)?,
                    artist: row.get(2)?,
                    length: row.get(3)?,
                    path: row.get(4)?
                }
            )
        }).unwrap();
        for song in song_iter {
            songs.push(song.unwrap());
            num_songs += 1;
        }

        //get the list of albums
        let mut cmd = conn.prepare("SELECT DISTINCT album FROM library ORDER BY album").unwrap();

        let album_iter = cmd.query_map([], |row|{
            Ok(SqlString{text: row.get(0)?})
        }).unwrap();

        for album in album_iter {
            album_names.push(album.unwrap().text);
            num_albums += 1;
        }

        let mut artist_name =  String::new();
        for name in album_names {
            //find the first song in the album to get the artist name
            for song in &songs {
                if song.album == name {
                    artist_name = song.artist.clone();
                }
            }
            //get the rest of the songs
            let mut album = Album::init(&name, &artist_name);
            for song in &songs {
                if song.album == name {
                    let song_to_push = song.clone();
                    
                    album.songs.push(song_to_push);
                }
            }
            albums.push(album);
        }

        //get the list of artists
        let mut cmd = conn.prepare("SELECT DISTINCT artist FROM library ORDER BY artist").unwrap();
        
        let artist_iter = cmd.query_map([], |row|{
            Ok(SqlString{text: row.get(0)?})
        }).unwrap();

        for artist in artist_iter {
            artist_names.push(artist.unwrap().text);
            num_artists += 1;
        }

        for name in artist_names {
            let mut artist = Artist::init(&name);

            for album in &albums {
                if album.artist == artist.name {
                    let album_to_push = album.clone();

                    artist.albums.push(album_to_push);
                }
            }
            artists.push(artist);
        }

        //construct the library database in memory
        Library {
            artists,
            num_songs,
            num_albums,
            num_artists
        }
    }
}

#[derive(Clone)]
pub struct Playlist {
    // the playlist's name (will be "_" if it is the play queue)
    pub name: String,

    // the songs that the playlist has
    pub songs: Vec<Song>
}

impl fmt::Display for Playlist {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        writeln!(f,"Playlist name: {}",self.name).unwrap();
        writeln!(f,"Songs:").unwrap();
        for song in &self.songs {
            writeln!(f, "Title: {}",song.name).unwrap();
            writeln!(f, "Album: {}",song.name).unwrap();
            writeln!(f, "Artist: {}",song.artist).unwrap();
        }
        Ok(())
    }
}

impl Playlist {
    pub fn new<N: Into<String>>(name: N) -> Playlist{
        Playlist {
            name: name.into(),
            songs: Vec::new()
        }
    }

    // add a song to the playlist
    pub fn add_song(&mut self, song: &Song) {
        self.songs.push(song.clone());
    }

    // add a whole album to the playlist
    pub fn add_album(&mut self, album: &Album) {
        for song in &album.songs {
            self.songs.push(song.clone());
        }
    }

    // add all of an artist's songs to the playlist
    pub fn add_artist(&mut self, artist: &Artist) {
        for album in &artist.albums {
            for song in &album.songs {
                self.songs.push(song.clone());
            }
        }
    }

    // shuffles the playlist
    pub fn shuffle(&mut self, first: usize) {
        let mut songs = self.songs.clone();
        let first_song = songs.remove(first);
        let length = songs.len();
        let mut rand = rand::thread_rng();
        let mut new_songs = vec![first_song];

        for _ in 0..length {
            let song = songs.remove(rand.gen_range(0..songs.len()));

            new_songs.push(song);
        }
        self.songs = new_songs;
    }
}
// generates the database used for the music library
// returns the database's connection handle
pub fn gendb() -> rusqlite::Connection {

    //the default music library path
    let library_path = format!("{}/Music",env::var("HOME").expect("Where the hell is your home folder?!"));

    //create the database
    let conn = Connection::open(format!("{}/.local/share/music-lounge/library.db",env::var("HOME").unwrap())).unwrap();
    
    //create the table for the library database
    conn.execute("CREATE TABLE library (name TEXT NOT NULL, album TEXT NOT NULL, artist TEXT NOT NULL, length INTEGER NOT NULL, path TEXT NOT NULL);",[]).unwrap();

    //get a recursive listing of all the music files in the music library
    let library_files = format!("{}/*/*/*.ogg",&library_path);
    let mut file_list: Vec<String> = Vec::new();
    for entry in glob(&library_files[..]).expect("Failed to read glob pattern") {
        file_list.push(format!("{}",entry.unwrap().display()));
    }

    //parse the metadata of each file
    for entry in file_list {
        let song_file = File::open(&entry).unwrap();
        let song_file2 = File::open(&entry).unwrap();
        let song = OggStreamReader::new(song_file).unwrap();
        let mut title = " ".to_string();
        let mut album = " ".to_string();
        let mut artist = " ".to_string();

        //get the title, album, and artist of the song
        for comment in song.comment_hdr.comment_list {
            match comment.0.as_str() {
                "TITLE"|"title" => title=comment.1,
                "ALBUM"|"album" => album=comment.1,
                "ARTIST"|"artist" => artist=comment.1,
                _=>{}
            }
        }

        //get the song duration
        let mut song_meta_vec = ogg_metadata::read_format(song_file2).unwrap();
        let song_meta = song_meta_vec.pop().unwrap();
        let metadata = match song_meta {
            ogg_metadata::OggFormat::Vorbis(meta) => meta,
            _=>{
                panic!("Unknown type!")
            }
        };
        let duration = metadata.get_duration().unwrap().as_secs_f32().ceil() as usize;

        conn.execute("INSERT INTO library VALUES (?1,?2,?3,?4,?5);",params![title,album,artist,duration,entry]).unwrap();
    }
    conn
}