
use std::pin::Pin;

use jsonrpc_core::Result;
use tokio::io::{AsyncWrite, AsyncWriteExt};

use super::RpcErrorCodes;

pub struct RpcChannelRecord {
    on_data_sink: Pin<Box<dyn AsyncWrite + Send + Sync + 'static>>,
}

impl RpcChannelRecord {
    pub fn new(
        on_data_sink: Pin<Box<dyn AsyncWrite + Send + Sync + 'static>>,
    ) -> Self {
        return RpcChannelRecord {
            on_data_sink,
        };
    }

    pub async fn send_data<'a>(&'a mut self, data: &'a [u8]) -> Result<usize> {
        let result = self.on_data_sink
            .write(&data[..]).await;

        return match result {
            Ok(bytes_sent) => Ok(bytes_sent),
            Err(error) => {
                Err(
                    RpcErrorCodes::ChannelStreamFailure(error.to_string()).into(),
                )
            },
        };
    }
}

#[cfg(test)]
mod tests {
    mod send_data_method {
        use cs_utils::random_str_rg;
        use rstest::rstest;
        use tokio::io::duplex;
        use crate::multiplexed_connection::rpc::services::rpc_channel_record::RpcChannelRecord;

        use tokio::try_join;

        #[rstest]
        #[case::size_8_32(8, 32)]
        #[case::size_128_512(128, 512)]
        #[case::size_2048_4096(2048, 4096)]
        #[case::size_4096_8192(4096, 8192)]
        #[case::size_8192_16384(8192, 16384)]
        #[tokio::test]
        async fn sends_data(
            #[case] str_min_size: usize,
            #[case] str_max_size: usize,
        ) {
            use cs_utils::random_number;
            use tokio::io::AsyncReadExt;

            let (mut source, sink) = duplex(str_min_size);

            let mut channel_record = RpcChannelRecord::new(Box::pin(sink));

            let test_data = vec![
                random_str_rg(str_min_size..=str_max_size),
                random_str_rg(str_min_size..=str_max_size),
                random_str_rg(str_min_size..=str_max_size),
                random_str_rg(str_min_size..=str_max_size),
                random_str_rg(str_min_size..=str_max_size),
                random_str_rg(str_min_size..=str_max_size),
                random_str_rg(str_min_size..=str_max_size),
            ].join("");

            let data_to_send = test_data.clone();

            try_join!(
                tokio::spawn(async move {
                    let mut i = 0;
                    let data = data_to_send.as_bytes();

                    while i < data_to_send.len() {
                        let message_len = random_number(str_min_size..=str_max_size) / 2;
                        
                        let message_len = if i + message_len < data.len() {
                            i + message_len
                        } else {
                            data.len()
                        };

                        let bytes_sent = channel_record
                            .send_data(&data[i..message_len]).await
                            .expect("Cannot send a message.");

                        assert!(
                            bytes_sent > 0,
                            "No bytes sent.",
                        );
                            
                        i += bytes_sent as usize;
                    }
                }),
                tokio::spawn(async move {
                    let mut received_data = String::new();
    
                    loop {
                        let mut buf = [0; 1024];

                        let bytes_read = source.read(&mut buf).await.unwrap();
    
                        let result = std::str::from_utf8(&buf[..bytes_read])
                            .expect("Cannot parse UTF8 string.")
                            .to_string();
    
                        received_data = format!("{}{}", received_data, result);
    
                        if received_data.len() == test_data.len() {
                            break;
                        }
                    };
    
                    assert_eq!(
                        received_data,
                        test_data,
                        "Sent and received data must match.",
                    );
                })
            ).unwrap();
        }
    }
}
