use std::fmt;
use tokio::io::{DuplexStream, split};

use anyhow::{Result, bail};

use connection_utils::TDataReader;

pub struct RemoteRpcChannelEvent {
    id: u16,
    label: String,
    data_source: Option<TDataReader>,
}

impl RemoteRpcChannelEvent {
    pub fn new(
        id: u16,
        label: String,
        data_duplex: DuplexStream,
    ) -> RemoteRpcChannelEvent {
        let (source, _sink) = split(data_duplex);

        return RemoteRpcChannelEvent {
            id,
            label,
            data_source: Some(Box::pin(source)),
        };
    }
    
    /// Gets `RpcChannel` id.
    pub fn id(&self) -> u16 {
        return self.id;
    }

    /// Gets `RpcChannel` label.
    pub fn label(&self) -> &String {
        return &self.label;
    }

    /// Takes `on_data` listener from the structs.
    pub fn on_data(&mut self) -> Result<TDataReader> {
        match self.data_source.take() {
            None => {
                // TODO: test
                bail!("Cannot get \"on_data\" subscriber, already taken?");
            },
            Some(on_data) => {
                return Ok(on_data);
            }
        };
    }

    /// Puts `on_data` listener back to the struct.
    pub fn off_data(&mut self, data_source: TDataReader) -> Result<()> {
        if self.data_source.is_some() {
            bail!("\"on_data\" already defined.");
        }

        self.data_source = Some(data_source);

        return Ok(());
    }
}

impl fmt::Debug for RemoteRpcChannelEvent {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        return f.debug_struct("RemoteRpcChannelEvent")
            .field("id", &self.id)
            .field("label", &self.label)
            .finish();
    }
}

#[cfg(test)]
mod tests {
    use cs_utils::{random_number, random_str_rg};
    use rstest::fixture;
    use tokio::io::{duplex, DuplexStream};
    
    pub use super::RemoteRpcChannelEvent;

    #[fixture]
    fn rpc_channel_event() -> (RemoteRpcChannelEvent, DuplexStream) {
        let (duplex1, duplex2) = duplex(4096);

        let rpc_channel_event = RemoteRpcChannelEvent::new(
            random_number(0..=u16::MAX),
            random_str_rg(8..=16),
            duplex1,
        );

        return (
            rpc_channel_event,
            duplex2,
        );
    }

    mod on_data {
        use rstest::rstest;
        use tokio::io::DuplexStream;

        use crate::multiplexed_connection::rpc::rpc_channel::remote_rpc_channel_event::tests::rpc_channel_event;

        use super::RemoteRpcChannelEvent;

        #[rstest]
        #[test]
        fn sets_on_data(
            rpc_channel_event: (RemoteRpcChannelEvent, DuplexStream),
        ) {
            let (
                mut rpc_channel_event,
                _duplex2,
            ) = rpc_channel_event;

            assert!(
                rpc_channel_event.on_data().is_ok(),
                "Must have \"on_data\" handler.",
            );

            assert!(
                rpc_channel_event.on_data().is_err(),
                "Must not have \"on_data\" handler.",
            );
        }
    }

    mod off_data {
        use cs_utils::{random_number, random_str_rg};
        use rstest::rstest;
        use tokio::io::{duplex, DuplexStream};

        use crate::multiplexed_connection::rpc::rpc_channel::remote_rpc_channel_event::tests::rpc_channel_event;

        use super::RemoteRpcChannelEvent;

        #[rstest]
        #[test]
        fn sets_on_data_back(
            rpc_channel_event: (RemoteRpcChannelEvent, DuplexStream),
        ) {
            let (
                mut rpc_channel_event,
                _duplex2,
            ) = rpc_channel_event;

            let on_data = rpc_channel_event.on_data()
                .unwrap();

            assert!(
                rpc_channel_event.on_data().is_err(),
                "Must not have \"on_data\" handler.",
            );

            rpc_channel_event.off_data(on_data)
                .unwrap();

            assert!(
                rpc_channel_event.on_data().is_ok(),
                "Must have \"on_data\" handler.",
            );
        }

        #[test]
        fn errors_if_already_set() {
            let (duplex1, duplex2) = duplex(4096);

            let mut rpc_channel_event = RemoteRpcChannelEvent::new(
                random_number(0..=u16::MAX),
                random_str_rg(8..=16),
                duplex1,
            );

            assert!(
                rpc_channel_event.off_data(Box::pin(duplex2)).is_err(),
                "Must return error if \non_data\n already set.",
            );
        }
    }

    mod implementations {
        use cs_utils::{random_number, random_str_rg, test::{implements_send, implements_debug}};
        use tokio::io::duplex;

        use super::RemoteRpcChannelEvent;

        #[test]
        fn implements_send_trait() {
            let (duplex1, _duplex2) = duplex(4096);

            let rpc_channel_event = RemoteRpcChannelEvent::new(
                random_number(0..=u16::MAX),
                random_str_rg(8..=16),
                duplex1,
            );

            implements_send(rpc_channel_event);
        }

        #[test]
        fn implements_debug_trait() {
            let (duplex1, _duplex2) = duplex(4096);

            let rpc_channel_event = RemoteRpcChannelEvent::new(
                random_number(0..=u16::MAX),
                random_str_rg(8..=16),
                duplex1,
            );

            implements_debug(rpc_channel_event);
        }
    }
}
