use jsonrpc_core::ErrorCode;
use jsonrpc_core::Error as JsonRpcError;

pub enum RpcErrorCodes {
    ChannelCreationFailed,
    ChannelRemovalFailed,
    NoChannelFound,
    ChannelStreamFailure(String),
}

impl Into<i64> for RpcErrorCodes {
    fn into(self) -> i64 {
        return match self {
            RpcErrorCodes::ChannelCreationFailed => -33000,
            RpcErrorCodes::ChannelRemovalFailed => -33010,
            RpcErrorCodes::NoChannelFound => -33020,
            RpcErrorCodes::ChannelStreamFailure(_) => -33030,
        };
    }
}

impl Into<ErrorCode> for RpcErrorCodes {
    fn into(self) -> ErrorCode {
        let error_code: i64 = self.into();

        return ErrorCode::ServerError(error_code);
    }
}

impl Into<JsonRpcError> for RpcErrorCodes {
    fn into(self) -> JsonRpcError {
        let error_code: ErrorCode = self.into();

        return JsonRpcError::new(error_code);
    }
}

impl From<ErrorCode> for RpcErrorCodes {
    fn from(error_code: ErrorCode) -> RpcErrorCodes {
        let error_code = match error_code {
            ErrorCode::ServerError(code) => code,
            unsupported @ _ => {
                panic!("Unsupported error code {:?}.", unsupported);
            },
        };

        return match error_code {
            -33000 => RpcErrorCodes::ChannelCreationFailed,
            -33010 => RpcErrorCodes::ChannelRemovalFailed,
            -33020 => RpcErrorCodes::NoChannelFound,
            -33030 => RpcErrorCodes::ChannelStreamFailure("Channel data send failed.".to_string()),
            unknown @ _ => {
                panic!("Unknown error code: {:?}.", unknown);
            },
        };
    }
}
