//! Represents the source of truth in the application
//!
//! Should pay special attention to properly model state in this module,
//! as it should prevent data anomalies

use crate::{
    input::TextInput,
    types::{self, Chat, ChatFocus, DirectChat, Focus, Mode, User},
};

pub struct State {
    pub mode: types::Mode,
    pub focus: types::Focus,
    pub chats: Chats,
    pub me: types::User,
    pub command: Option<TextInput>,
}

impl State {
    pub fn new() -> Self {
        Self {
            mode: Mode::Normal,
            focus: Focus::Chat(ChatFocus::Input),
            chats: Chats {
                chats: vec![(
                    Chat::Direct(DirectChat {
                        member: User {
                            name: String::from("qnbst"),
                        },
                        messages: vec![],
                        input: TextInput::new(),
                    }),
                    true,
                )],
            },
            me: User {
                name: String::from("parasrah"),
            },
            command: None,
        }
    }
}

impl State {
    pub fn get_focused_input_mut(&mut self) -> Option<&mut TextInput> {
        match (&self.mode, &self.focus) {
            (Mode::Command, _) => self.command.as_mut(),
            (_, Focus::Chat(ChatFocus::Input)) => Some(self.chats.get_active_mut().unwrap().input_mut()),
            _ => None,
        }
    }
}

pub struct Chats {
    chats: Vec<(types::Chat, bool)>,
}

impl Chats {
    pub fn get_active(&self) -> Option<&types::Chat> {
        self.chats
            .iter()
            .find(|(_, active)| *active)
            .map(|(x, _)| x)
    }

    pub fn get_active_mut(&mut self) -> Option<&mut types::Chat> {
        self.chats
            .iter_mut()
            .find(|(_, active)| *active)
            .map(|(x, _)| x)
    }

    pub fn move_up(&mut self) {
        todo!("move_up")
    }

    pub fn move_down(&mut self) {
        todo!("move_down")
    }

    pub fn collect(&self) -> Vec<&types::Chat> {
        self.chats.iter().map(|(x, _)| x).collect()
    }
}
