use std::io;

use futures_util::{FutureExt, StreamExt};
use tokio::select;

use crossterm::event::EventStream;
use crossterm::{
    event::{DisableMouseCapture, EnableMouseCapture},
    execute,
    terminal::{disable_raw_mode, enable_raw_mode, EnterAlternateScreen, LeaveAlternateScreen},
};
use tui::{
    backend::{Backend, CrosstermBackend},
    Terminal,
};

use mtrx_rs::{handle_terminal_event, paint, Action, State};

async fn event_loop<B: Backend>(
    state: &mut State,
    terminal: &mut Terminal<B>,
) -> Result<(), io::Error> {
    let mut terminal_reader = EventStream::new();
    loop {
        let terminal_event = terminal_reader.next().fuse();

        select! {
            maybe_event = terminal_event => {
                match maybe_event {
                    Some(Ok(event)) => {
                        let actions = handle_terminal_event(event, state);
                        if actions.contains(&Action::Terminate) {
                            break
                        }
                        if actions.contains(&Action::Repaint) {
                            terminal.draw(|f| paint(&state, f.size(), f))?;
                        }
                    },
                    Some(Err(_)) => todo!("handle error"),
                    None => break
                }
            },
        }
    }

    Ok(())
}

#[tokio::main]
async fn main() -> Result<(), io::Error> {
    // initialize terminal
    enable_raw_mode()?;
    let mut stdout = io::stdout();
    execute!(stdout, EnterAlternateScreen, EnableMouseCapture)?;
    let backend = CrosstermBackend::new(stdout);
    let mut terminal = Terminal::new(backend)?;
    let mut state = State::new();
    terminal.draw(|f| paint(&state, f.size(), f))?;

    // run event loop
    event_loop(&mut state, &mut terminal).await?;

    // restore terminal
    disable_raw_mode()?;
    execute!(
        terminal.backend_mut(),
        LeaveAlternateScreen,
        DisableMouseCapture
    )?;
    terminal.show_cursor()?;

    Ok(())
}
