//! This module models the interface state, which is derived from
//! the application state, and should **not** be mutated

use tui::backend::Backend;
use tui::layout::{Constraint, Direction, Layout, Rect};
use tui::style::{Color, Modifier, Style};
use tui::widgets::{Block, Borders, List, ListItem, Paragraph};
use tui::Frame;

use crate::input::TextInput;
use crate::state::{Chats, State};
use crate::types::{self, Chat, ChatFocus, DirectChat, Focus, GroupChat, Message, Mode};

pub fn paint<B: Backend>(state: &State, rect: Rect, f: &mut Frame<B>) {
    let all_chunks = Layout::default()
        .direction(Direction::Vertical)
        .constraints([Constraint::Min(10), Constraint::Length(1)])
        .split(rect);
    let top_chunks = Layout::default()
        .direction(Direction::Horizontal)
        .constraints([Constraint::Percentage(10), Constraint::Percentage(90)].as_ref())
        .split(all_chunks[0]);

    let sidebar_view = build_sidebar_view(&state.chats, &state.focus);

    f.render_widget(sidebar_view, top_chunks[0]);

    match state.chats.get_active() {
        Some(chat) => paint_chat(f, top_chunks[1], chat, &state.mode, &state.focus),
        None => {}
    };

    paint_status_bar(f, all_chunks[1], state);
}

fn paint_chat<B: Backend>(
    f: &mut Frame<B>,
    rect: Rect,
    chat: &types::Chat,
    mode: &Mode,
    focus: &Focus,
) {
    let chunks = Layout::default()
        .direction(Direction::Vertical)
        .constraints([Constraint::Min(10), Constraint::Length(3)].as_ref())
        .split(rect);

    match chat {
        Chat::Direct(DirectChat {
            messages, input, ..
        }) => {
            let messages_view = build_messages_view(messages, focus);
            let input_view = build_input_view(input, focus);
            f.render_widget(messages_view, chunks[0]);
            f.render_widget(input_view, chunks[1]);
            if let (Mode::Insert, Focus::Chat(ChatFocus::Input)) = (mode, focus) {
                let x: u16 = chunks[1].x + u16::from(input.cursor()) + 1;
                let y: u16 = chunks[1].y + 1;
                f.set_cursor(x, y)
            }
        }
        Chat::Group(_) => todo!("group chat"),
    };
}

fn paint_status_bar<B: Backend>(
    f: &mut Frame<B>,
    rect: Rect,
    state: &State
) {
    if let (Some(command), Mode::Command) = (&state.command, &state.mode) {
        let command_view = build_command_view(command);
        f.render_widget(command_view, rect);
        let x: u16 = rect.x + u16::from(command.cursor()) + 1;
        let y: u16 = rect.y;
        f.set_cursor(x, y);
    }
}

fn build_command_view<'a>(command: &'a TextInput) -> Paragraph<'a> {
    let command_txt = format!(":{}", command.get_content());
    Paragraph::new(command_txt)
}

fn build_sidebar_view<'a>(chats: &Chats, focus: &Focus) -> List<'a> {
    let sidebar_view_items: Vec<ListItem> = chats
        .collect()
        .iter()
        .map(|chat| match chat {
            Chat::Direct(DirectChat { member, .. }) => ListItem::new(member.name.clone()),
            Chat::Group(GroupChat { group_name, .. }) => ListItem::new(group_name.clone()),
        })
        .collect();

    let has_focus = match focus {
        Focus::Sidebar => true,
        _ => false,
    };

    let border = build_border(None, has_focus);

    List::new(sidebar_view_items)
        .block(border)
        .style(Style::default().fg(Color::White))
        .highlight_style(Style::default().add_modifier(Modifier::ITALIC))
        .highlight_symbol(">>")
}

fn build_input_view<'a>(input: &'a TextInput, focus: &Focus) -> Paragraph<'a> {
    let has_focus = match focus {
        Focus::Chat(ChatFocus::Input) => true,
        _ => false,
    };

    let border = build_border(None, has_focus);

    Paragraph::new(input.get_content()).block(border)
}

fn build_messages_view<'a>(messages: &Vec<Message>, focus: &Focus) -> List<'a> {
    let items: Vec<ListItem> = messages
        .iter()
        .map(|message| ListItem::new(message.content.clone()))
        .collect();

    let has_focus = match focus {
        Focus::Chat(ChatFocus::Message(_)) => true,
        _ => false,
    };

    let border = build_border(None, has_focus);

    List::new(items).block(border)
}

fn build_border<'a>(title: Option<String>, has_focus: bool) -> Block<'a> {
    let border_style = if has_focus {
        Style::default().fg(Color::White)
    } else {
        Style::default().fg(Color::Gray)
    };
    let mut block = Block::default()
        .borders(Borders::ALL)
        .border_style(border_style);

    if let Some(title) = title {
        block = block.title(title);
    }

    block
}
