use crossterm::event::{
    Event as TerminalEvent,
    KeyCode::{self, Char},
    KeyEvent, KeyModifiers as Mod,
};

use crate::{
    input::TextInput,
    state::State,
    types::{ChatFocus, Focus, Mode},
};

#[derive(PartialEq)]
pub enum Action {
    Terminate,
    Repaint,
    SendMessage(String),
}

impl Action {
    fn one(self) -> Vec<Self> {
        vec![self]
    }
}

pub fn handle_terminal_event(event: TerminalEvent, state: &mut State) -> Vec<Action> {
    match event {
        TerminalEvent::Key(event) => match event {
            KeyEvent { code, modifiers } => match (&state.mode, &state.focus, code, modifiers) {
                (_, _, Char('c'), Mod::CONTROL) => Action::Terminate.one(),

                // -- modes --
                (Mode::Normal, Focus::Chat(ChatFocus::Input), Char('i'), Mod::NONE) => {
                    state.mode = Mode::Insert;
                    Action::Repaint.one()
                }
                (Mode::Insert, _, KeyCode::Esc, Mod::NONE) => {
                    state.mode = Mode::Normal;
                    Action::Repaint.one()
                }
                (Mode::Normal, _, Char(':'), _) => {
                    state.command = Some(TextInput::new());
                    state.mode = Mode::Command;
                    Action::Repaint.one()
                }
                (Mode::Command, _, KeyCode::Esc, Mod::NONE) => {
                    state.command = None;
                    state.mode = Mode::Normal;
                    Action::Repaint.one()
                }

                // -- normal mode --
                (Mode::Normal, _, Char('q'), _) => Action::Terminate.one(),
                (Mode::Normal, Focus::Sidebar, Char('j'), Mod::SHIFT) => {
                    state.chats.move_down();
                    Action::Repaint.one()
                }
                (Mode::Normal, Focus::Sidebar, Char('k'), Mod::SHIFT) => {
                    state.chats.move_up();
                    Action::Repaint.one()
                }

                // focus
                (Mode::Normal, Focus::Sidebar, Char('L'), Mod::SHIFT) => {
                    state.focus = Focus::Chat(ChatFocus::Input);
                    Action::Repaint.one()
                }
                (Mode::Normal, Focus::Chat(_), Char('H'), Mod::SHIFT) => {
                    state.focus = Focus::Sidebar;
                    Action::Repaint.one()
                }
                (Mode::Normal, Focus::Chat(ChatFocus::Message(_)), Char('J'), Mod::SHIFT) => {
                    state.focus = Focus::Chat(ChatFocus::Input);
                    Action::Repaint.one()
                }
                (Mode::Normal, Focus::Chat(ChatFocus::Input), Char('K'), Mod::SHIFT) => {
                    state.focus = Focus::Chat(ChatFocus::Message(None));
                    Action::Repaint.one()
                }

                // -- insert/command mode --
                (Mode::Insert, _, KeyCode::Enter, _) => match state.get_focused_input_mut() {
                    Some(input) => {
                        input.clear();
                        vec![
                            Action::Repaint,
                            Action::SendMessage(input.get_content().into()),
                        ]
                    }
                    None => vec![],
                },
                (
                    Mode::Insert | Mode::Command,
                    Focus::Chat(ChatFocus::Input),
                    Char(char),
                    Mod::NONE | Mod::SHIFT,
                ) => match state.get_focused_input_mut() {
                    Some(input) => {
                        input.add_char(&char);
                        Action::Repaint.one()
                    }
                    None => vec![],
                },
                (
                    Mode::Insert | Mode::Command,
                    Focus::Chat(ChatFocus::Input),
                    KeyCode::Backspace,
                    Mod::NONE,
                ) => match state.get_focused_input_mut() {
                    Some(input) => {
                        input.delete_char();
                        Action::Repaint.one()
                    }
                    None => vec![],
                },
                (
                    Mode::Insert | Mode::Command,
                    Focus::Chat(ChatFocus::Input),
                    Char('b'),
                    Mod::CONTROL,
                ) => match state.get_focused_input_mut() {
                    Some(input) => {
                        input.move_left_char();
                        Action::Repaint.one()
                    }
                    None => vec![],
                },
                (
                    Mode::Insert | Mode::Command,
                    Focus::Chat(ChatFocus::Input),
                    Char('f'),
                    Mod::CONTROL,
                ) => match state.get_focused_input_mut() {
                    Some(input) => {
                        input.move_right_char();
                        Action::Repaint.one()
                    }
                    None => vec![],
                },
                _ => vec![],
            },
        },
        TerminalEvent::Mouse(_event) => vec![],
        TerminalEvent::Resize(_x, _y) => Action::Repaint.one(),
    }
}
