use std::fmt;
use std::result;

pub type Result<T> = result::Result<T, Error>;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct Error {
    msg: String,
}

impl Error {
    pub fn new<S: AsRef<str>>(msg: S) -> Self {
        Error {
            msg: msg.as_ref().to_string(),
        }
    }

    pub fn as_err<S: AsRef<str>, T>(msg: S) -> Result<T> {
        Error::new(msg.as_ref()).to_err()
    }

    pub fn to_err<T>(&self) -> Result<T> {
        Err(self.clone())
    }

    pub fn with_title(&self, title: &str) -> Self {
        Error::new(format!("{}: {}", title, self.msg))
    }

    pub unsafe fn panic(&self) -> ! {
        panic!("{}", self.msg);
    }
}

impl fmt::Display for Error {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.msg)
    }
}
