use super::*;

pub type TestError = Error;

#[test]
fn test_err() {
    let err = TestError::new("test error");

    assert_eq!(err.to_string(), "test error");
    assert_eq!(format!("{}", err), "test error");
}

#[test]
fn test_io() {
    let mut app = App::default();
    let title = Title::new(&mut app, "test_io");

    let mut io = IOHandle::new(&title);

    match io.write(&Stdin, "test write") {
        Ok(_) => panic!("write should fail"),
        Err(err) => assert_eq!(err.to_string(), "stdin not supported"),
    };

    match io.write(&Stdout, "test write") {
        Ok(ok) => assert_eq!(ok, 10),
        Err(err) => panic!("{}", err.to_string()),
    };

    match io.write(&Stderr, "test write") {
        Ok(ok) => assert_eq!(ok, 10),
        Err(err) => panic!("{}", err.to_string()),
    };

    match io.read(&Stdout) {
        Ok(ok) => assert_eq!(ok, "test write"),
        Err(err) => panic!("{}", err.to_string()),
    };

    match io.read(&Stderr) {
        Ok(ok) => assert_eq!(ok, "test write"),
        Err(err) => panic!("{}", err.to_string()),
    };

    match io.flush(&Stdin) {
        Ok(_) => panic!("flush should fail"),
        Err(err) => assert_eq!(err.to_string(), "stdin not supported"),
    };

    match io.flush(&Stdout) {
        Ok(ok) => assert_eq!(ok, 0),
        Err(err) => panic!("{}", err.to_string()),
    };

    match io.flush(&Stderr) {
        Ok(ok) => assert_eq!(ok, 0),
        Err(err) => panic!("{}", err.to_string()),
    };

    match io.flush_all() {
        Ok(ok) => assert_eq!(ok, 0),
        Err(err) => panic!("{}", err.to_string()),
    }

    assert_eq!(io.title().to_string(), "test_io");
}

#[test]
fn test_fmt() {
    use colored::Colorize;

    match format_str(&FormatType::Plain, "test", "format") {
        Ok(ok) => assert_eq!(ok, "test: format"),
        Err(err) => panic!("{}", err.to_string()),
    };

    match format_str(&FormatType::Success, "test", "format") {
        Ok(ok) => assert_eq!(ok, format!("{} format", "        test".green().bold())),
        Err(err) => panic!("{}", err.to_string()),
    };

    match format_str(&FormatType::Error, "test", "format") {
        Ok(ok) => assert_eq!(ok, format!("{} format", "        test".red().bold())),
        Err(err) => panic!("{}", err.to_string()),
    };

    match format_str(&FormatType::Warning, "test", "format") {
        Ok(ok) => assert_eq!(ok, format!("{} format", "        test".yellow().bold())),
        Err(err) => panic!("{}", err.to_string()),
    };
}

#[test]
fn test_ui() {
    #[allow(unused_mut, unused_variables)]
    fn test_app() {
        let mut app = App::default();
    }

    fn test_title() {
        let mut app = App::default();
        let title = Title::new(&mut app, "test_ui");

        assert_eq!(title.to_string(), "test_ui");
    }

    test_app();
    test_title();
}
