use super::*;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub enum FormatType {
    Plain,
    Success,
    Error,
    Warning,
}

impl FormatType {
    pub fn new() -> Self {
        FormatType::Plain
    }
}

pub fn indent<S: AsRef<str>>(string: S) -> Result<String, IndentError> {
    let len: i128 = 12 - string.as_ref().len() as i128;

    if len < 0 {
        Err(IndentError::new(format!("12 - string len < 0")))
    } else {
        let mut res = String::new();

        for _ in 0..len {
            res.push(' ');
        }

        res.push_str(string.as_ref());

        Ok(res)
    }
}

pub fn format_str<S: AsRef<str>>(
    type_: &FormatType,
    title: S,
    string: S,
) -> Result<String, FormatError> {
    use colored::Colorize;

    match type_ {
        FormatType::Plain => Ok(format!("{}: {}", title.as_ref(), string.as_ref())),
        FormatType::Success => Ok(format!(
            "{} {}",
            match indent(title) {
                Ok(ok) => ok.green().bold(),
                Err(err) =>
                    return Err(FormatError::new(format!(
                        "indent error: {}",
                        err.to_string()
                    ))),
            },
            string.as_ref()
        )),
        FormatType::Error => Ok(format!(
            "{} {}",
            match indent(title) {
                Ok(ok) => ok.red().bold(),
                Err(err) =>
                    return Err(FormatError::new(format!(
                        "indent error: {}",
                        err.to_string()
                    ))),
            },
            string.as_ref()
        )),
        FormatType::Warning => Ok(format!(
            "{} {}",
            match indent(title) {
                Ok(ok) => ok.yellow().bold(),
                Err(err) =>
                    return Err(FormatError::new(format!(
                        "indent error: {}",
                        err.to_string()
                    ))),
            },
            string.as_ref()
        )),
    }
}

pub type IndentError = Error;
pub type FormatError = Error;
