use super::*;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub enum IOType {
    Stdin,
    Stdout,
    Stderr,
    Null
}

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct IOHandle {
    title: Title,
    stdin: String,
    stdout: String,
    stderr: String
}

impl IOHandle {
    pub fn new(title: &Title) -> IOHandle {
        IOHandle {
            title: title.clone(),
            stdin: String::new(),
            stdout: String::new(),
            stderr: String::new()
        }
    }

    pub fn write<S: AsRef<str>>(&mut self, type_: &IOType, msg: S) -> Result<usize, IOError> {
        let msg = msg.as_ref().to_string();

        match type_ {
            &IOType::Stdin => {
                Err(IOError::new("stdin not supported"))
            },
            &IOType::Stdout => {
                self.stdout.push_str(&msg);

                Ok(msg.len())
            },
            &IOType::Stderr => {
                self.stderr.push_str(&msg);

                Ok(msg.len())
            },
            &IOType::Null => {
                Ok(msg.len())
            }
        }
    }

    pub fn read(&self, type_: &IOType) -> Result<String, IOError> {
        use std::io;
        
        let mut res = String::new();

        match type_ {
            &IOType::Stdin => {
                match io::stdin().read_line(&mut res) {
                    Ok(_) => Ok(res),
                    Err(err) => Err(IOError::new(&format!("{}", err)))
                }
            },
            &IOType::Stdout => {
                res.push_str(&self.stdout);

                Ok(res)
            },
            &IOType::Stderr => {
                res.push_str(&self.stderr);

                Ok(res)
            },
            &IOType::Null => {
                Ok(res)
            }
        }
    }

    pub fn flush(&mut self, type_: &IOType) -> Result<usize, IOError> {
        use std::io::{self, Write};

        match type_ {
            &IOType::Stdin => {
                Err(IOError::new("stdin not supported"))
            },
            &IOType::Stdout => {
                let copy = self.stdout.clone();

                let mut stdout = io::stdout();

                match stdout.write(copy.as_bytes()) {
                    Ok(_) => (),
                    Err(err) => return Err(IOError::new(&format!("{}", err)))
                }

                self.stdout.clear();

                Ok(self.stdout.len())
            },
            &IOType::Stderr => {
                let copy = self.stderr.clone();
                
                let mut stderr = io::stderr();

                match stderr.write(copy.as_bytes()) {
                    Ok(_) => (),
                    Err(err) => return Err(IOError::new(&format!("{}", err)))
                }

                self.stderr.clear();

                Ok(self.stderr.len())
            },
            &IOType::Null => {
                Ok(0)
            }
        }
    }

    pub fn flush_all(&mut self) -> Result<usize, IOError> {
        let mut res = 0;

        res += self.flush(&IOType::Stdout)?;
        res += self.flush(&IOType::Stderr)?;
    
        Ok(res)
    }
}

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct IOError {
    msg: String
}

impl Error for IOError {
    fn new<S: AsRef<str>>(msg: S) -> Self {
        IOError { msg: msg.as_ref().to_string() }
    }

    fn to_string(&self) -> String {
        self.msg.clone()
    }
}
