use super::*;
use std::io::{Write, stderr, stdout, stdin};

#[derive(Clone)]
pub struct CLI {
    title: String,
    history: Vec<String>
}

impl CLI {
    pub fn new<S: AsRef<str>>(title: S) -> CLI {
        CLI {
            title: title.as_ref().to_string(),
            history: Vec::new()
        }
    }

    pub fn println<S: AsRef<str>>(&mut self, msg: S) {
        println!("[{}] {}", self.title, msg.as_ref().to_string());
        match stdout().flush() {
            Ok(ok) => ok,
            Err(_err) => ()
        };

        self.history.push(msg.as_ref().to_string());
    }

    pub fn eprintln<S: AsRef<str>>(&mut self, msg: S) {
        eprintln!("[{} \x1B[31;1merror\x1B[0;0m] {}", self.title, msg.as_ref().to_string());
        match stderr().flush() {
            Ok(ok) => ok,
            Err(_err) => ()
        };

        self.history.push(msg.as_ref().to_string());
    }

    pub fn read<S: AsRef<str>>(&mut self, prompt: S) -> Result<String, InputError> {
        print!("[{}] {}", self.title, prompt.as_ref().to_string());

        let mut buff = String::new();

        match stdin().read_line(&mut buff) {
            Ok(_ok) => Ok(buff.clone()),
            Err(err) => Err(InputError::from(err.to_string()))
        }
    }
    
    pub fn clone_history(&self) -> Vec<String> {
        self.history.clone()
    }
}
