`mt_logger` is a multithreaded Rust logging library focused on traceability, and ease-of-use via macros.

Logs are stored in a `logs` directory, located inside the current working directory when a program is launched. The directory will be created if it does not already exist. Log file names conform to [ISO 8601](https://en.wikipedia.org/wiki/ISO_8601), with the exception that `:` is replaced with `_` to meet Windows file naming requirements. By default, the package name (pulled from `CARGO_PKG_NAME` environment variable) is prepended to the log file name.  
Example: `cool_game_engine/logs/cool_game_engine_2021-06-27T22_08_38.474-0600.log`

At initialization, a thread is created to receive log messages and commands from the main thread. Timestamps are set before sending in order to maintain complete traceability.

# Usage
The recommended method for using `mt_logger` is via macros. A global log sender/receiver pair is created by `mt_new!()`, so all further log messages and commands can be issued simply by calling the appropriate macro, e.g., `mt_log!()` to send a log message. No passing of references to a logger instance, or cloning of an `mpsc::Sender` required!

Additionally, all macros are designed to essentially no-op when the global instance is not initialized. This allows logging to be enabled/disabled with minimal code impact. For example, a program may be designed to only call `mt_new!()` when a `-log` switch is passed in, allowing the program to run silently when the switch is omitted.

### Note
Though accurate timestamps and correct ordering of messages are guaranteed, due to the nature of multithreading, the time at which a log message is recorded to an output stream is not. The `mt_flush!()` macro addresses this issue. It will block until all queued messages in the channel are flushed to the specified output stream(s). It is recommended that this macro be called during the shutdown of a program, otherwise any queued messages will be lost.

## Example

```rust
use mt_logger::*;

fn main() {
    // Initialize the mt_logger global instance
    mt_new!(None, Level::Info, OutputStream::Both);

    // Send a log message that WILL be output to BOTH file and stdout
    mt_log!(Level::Info, "Message {}: an INFO message", 1);
    // Send a log message that WILL NOT be output
    mt_log!(Level::Debug, "Message {}: a DEBUG message", 2);

    // Change the output stream to stdout only
    mt_stream!(OutputStream::StdOut);

    // Change the logging level
    mt_level!(Level::Trace);

    // Send a log message that WILL be output to stdout ONLY
    mt_log!(Level::Info, "Message {}: an INFO message", 3);
    // Send a log message that WILL be output to stdout ONLY
    mt_log!(Level::Trace, "Message {}: a TRACE message", 4);

    // Flush to ensure all messages reach the specified output
    mt_flush!().unwrap();

    // Get a count of the number of log messages
    let msg_count = mt_count!();
    println!("Messages logged: {}", msg_count);
}
```

## Output Samples

### Console
<img src="data:image/png;base64,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">

### File
```
2021-06-27T21:09:21.221721100: [  TRACE  ] mt_logger::tests::format_verification() line 597:
   This is a TRACE message.
2021-06-27T21:09:21.334663300: [  DEBUG  ] mt_logger::tests::format_verification() line 598:
   This is a DEBUG message.
2021-06-27T21:09:21.334673300: [  INFO   ] mt_logger::tests::format_verification() line 599:
   This is an INFO message.
2021-06-27T21:09:21.334677300: [ WARNING ] mt_logger::tests::format_verification() line 600:
   This is a WARNING message.
2021-06-27T21:09:21.334680800: [  ERROR  ] mt_logger::tests::format_verification() line 601:
   This is an ERROR message.
2021-06-27T21:09:21.334684300: [  FATAL  ] mt_logger::tests::format_verification() line 602:
   This is a FATAL message.
```