#![deny(warnings)]
use mt_dom::{patch::*, *};

pub type MyNode =
    Node<&'static str, &'static str, &'static str, &'static str, &'static str>;

#[test]
fn test_replace_node() {
    let old: MyNode = element("div", vec![], vec![]);
    let new = element("span", vec![], vec![]);

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(
        diff,
        vec![Patch::replace_node(
            Some(&"div"),
            TreePath::new(vec![]),
            &new
        )],
    );
}

#[test]
fn test_replace_text_node() {
    let old: MyNode = leaf("hello");
    let new = element("span", vec![], vec![]);

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(
        diff,
        vec![Patch::replace_node(None, TreePath::new(vec![]), &new)],
    );
}

#[test]
fn test_replace_node_in_child() {
    let old: MyNode =
        element("main", vec![], vec![element("div", vec![], vec![])]);
    let new = element("main", vec![], vec![element("span", vec![], vec![])]);

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(
        diff,
        vec![Patch::replace_node(
            Some(&"div"),
            TreePath::new(vec![0]),
            &element("span", vec![], vec![])
        )],
        "Should replace the first node"
    );
}

#[test]
fn test_205() {
    let old: MyNode = element(
        "div",
        vec![],
        vec![
            element(
                "b",
                vec![],
                vec![
                    element("i", vec![], vec![]),
                    element("i", vec![], vec![]),
                ],
            ),
            element("b", vec![], vec![]),
        ],
    ); //{ <div> <b> <i></i> <i></i> </b> <b></b> </div> },

    assert_eq!(5, old.node_count());
    let new = element(
        "div",
        vec![],
        vec![
            element("b", vec![], vec![element("i", vec![], vec![])]),
            element("i", vec![], vec![]),
        ],
    ); //{ <div> <b> <i></i> </b> <i></i> </div>},
    assert_eq!(
        dbg!(diff_with_key(&old, &new, &"key")),
        vec![
            Patch::remove_node(Some(&"i"), TreePath::new(vec![0, 1]),),
            Patch::replace_node(
                Some(&"b"),
                TreePath::new(vec![1]),
                &element("i", vec![], vec![])
            ),
        ],
    )
}

#[test]
fn test_no_changed() {
    let old: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![],
    );

    let new: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![],
    );

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(diff, vec![])
}

#[test]
fn test_attribute_order_changed() {
    let old: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![],
    );

    let new: MyNode = element(
        "div",
        vec![attr("class", "some-class"), attr("id", "some-id")],
        vec![],
    );

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(diff, vec![])
}

#[test]
fn test_class_changed() {
    let old: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![],
    );

    let new = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class2")],
        vec![],
    );

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(
        diff,
        vec![Patch::add_attributes(
            &"div",
            TreePath::new(vec![]),
            vec![&attr("class", "some-class2")]
        )]
    )
}

#[test]
fn leaf_node_changed() {
    let old: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![leaf("text1")],
    );

    let new = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![leaf("text2")],
    );

    let diff = diff_with_key(&old, &new, &"key");
    dbg!(&diff);
    assert_eq!(
        diff,
        vec![Patch::replace_node(
            None,
            TreePath::new(vec![0]),
            &leaf("text2")
        )]
    )
}

#[test]
fn test_class_will_not_be_merged_on_different_calls() {
    let old: MyNode = element("div", vec![], vec![]);

    let new = element(
        "div",
        vec![attr("class", "class1"), attr("class", "class2")],
        vec![],
    );

    let diff = diff_with_key(&old, &new, &"key");
    assert_ne!(
        diff,
        vec![Patch::add_attributes(
            &"div",
            TreePath::new(vec![]),
            vec![&Attribute::with_multiple_values(
                None,
                "class",
                vec!["class1", "class2"]
            )]
        )]
    )
}

#[test]
fn test_class_removed() {
    let old: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![],
    );

    let new = element("div", vec![attr("id", "some-id")], vec![]);

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(
        diff,
        vec![Patch::remove_attributes(
            &"div",
            TreePath::new(vec![]),
            vec![&attr("class", "some-class")]
        )]
    )
}

#[test]
fn test_multiple_calls_to_style() {
    let old: MyNode = element(
        "div",
        vec![
            attr("style", "display:flex"),
            attr("style", "width:100px;height:100px"),
        ],
        vec![],
    );

    let new = element(
        "div",
        vec![
            attr("style", "display:flex"),
            attr("style", "width:200px;height:200px"),
        ],
        vec![],
    );

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(
        diff,
        vec![Patch::add_attributes(
            &"div",
            TreePath::new(vec![]),
            vec![
                &attr("style", "display:flex"),
                &attr("style", "width:200px;height:200px"),
            ]
        )]
    )
}

#[test]
fn inner_html_func_calls() {
    let old: MyNode = element("div", vec![], vec![]);

    let new: MyNode =
        element("div", vec![attr("inner_html", "<h1>Hello</h2>")], vec![]);

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(
        diff,
        vec![Patch::add_attributes(
            &"div",
            TreePath::new(vec![]),
            vec![&attr("inner_html", "<h1>Hello</h2>")]
        )]
    )
}

#[test]
fn test_append() {
    let old: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![element("div", vec![], vec![leaf("1")])],
    );

    let new: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![
            element("div", vec![], vec![leaf("1")]),
            element("div", vec![], vec![leaf("2")]),
        ],
    );

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(
        diff,
        vec![Patch::append_children(
            &"div",
            TreePath::new(vec![]),
            vec![&element("div", vec![], vec![leaf("2")])],
        )]
    )
}

#[test]
fn test_append_more() {
    let old: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![element("div", vec![], vec![leaf("1")])],
    );

    let new: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![
            element("div", vec![], vec![leaf("1")]),
            element("div", vec![], vec![leaf("2")]),
            element("div", vec![], vec![leaf("3")]),
        ],
    );

    let diff = diff_with_key(&old, &new, &"key");
    assert_eq!(
        diff,
        vec![Patch::append_children(
            &"div",
            TreePath::new(vec![]),
            vec![
                &element("div", vec![], vec![leaf("2")]),
                &element("div", vec![], vec![leaf("3")])
            ],
        )]
    )
}

#[test]
fn test_append_at_sub_level() {
    let old: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![element(
            "main",
            vec![],
            vec![element("div", vec![], vec![leaf("1")])],
        )],
    );

    let new: MyNode = element(
        "div",
        vec![attr("id", "some-id"), attr("class", "some-class")],
        vec![element(
            "main",
            vec![],
            vec![
                element("div", vec![], vec![leaf("1")]),
                element("div", vec![], vec![leaf("2")]),
                element("div", vec![], vec![leaf("3")]),
            ],
        )],
    );

    let diff = diff_with_key(&old, &new, &"key");
    dbg!(&diff);
    assert_eq!(
        diff,
        vec![Patch::append_children(
            &"main",
            TreePath::new(vec![0]),
            vec![
                &element("div", vec![], vec![leaf("2")]),
                &element("div", vec![], vec![leaf("3")])
            ],
        )]
    )
}
