# mpi-fork-fnsp

Message Passing Interface bindings for Rust

The [Message Passing Interface][MPI] (MPI) is a specification for a
message-passing style concurrency library. Implementations of MPI are often used to structure
parallel computation on High Performance Computing systems. The MPI specification describes
bindings for the C programming language (and through it C++) as well as for the Fortran
programming language. This library tries to bridge the gap into a more rustic world.

[MPI]: http://www.mpi-forum.org


This crate is forked from [RSMPI](https://github.com/rsmpi/rsmpi).

## Main differences to original library
- Uses most recent version, older version on crates.io does not compile
- Add support for complex numbers
- Merge workspace into a single crate

## Usage

Add the `mpi` crate as a dependency in your `Cargo.toml`:

```toml
[dependencies]
mpi = { path = "mpi-fork-fnsp",  version = "0.6" }
```

Then use it in your program like this:

```rust
extern crate mpi;

use mpi::traits::*;

fn main() {
    let universe = mpi::initialize().unwrap();
    let world = universe.world();
    let size = world.size();
    let rank = world.rank();

    if size != 2 {
        panic!("Size of MPI_COMM_WORLD must be 2, but is {}!", size);
     }

    match rank {
        0 => {
            let msg = vec![4.0f64, 8.0, 15.0];
            world.process_at_rank(rank + 1).send(&msg[..]);
        }
        1 => {
            let (msg, status) = world.any_process().receive_vec::<f64>();
            println!("Process {} got message {:?}.\nStatus is: {:?}", rank, msg, status);
        }
        _ => unreachable!()
    }
}
```

## Features

The bindings follow the MPI 3.1 specification.

Currently supported:

- **Groups, Contexts, Communicators**:
  - Group and (Intra-)Communicator management from section 6 is mostly complete.
  - no Inter-Communicators
  - no process topologies
- **Point to point communication**:
  - standard, buffered, synchronous and ready mode send in blocking and non-blocking variants
  - receive in blocking and non-blocking variants
  - send-receive
  - probe
  - matched probe/receive
- **Collective communication**:
  - barrier
  - broadcast
  - (all) gather
  - scatter
  - all to all
  - varying counts operations
  - reductions/scans
  - blocking and non-blocking variants
- **Datatypes**: Bridging between Rust types and MPI basic types as well as custom MPI datatypes
which can act as views into buffers.

Not supported (yet):

- Process management
- One-sided communication (RMA)
- MPI parallel I/O
- A million small things

The sub-modules contain a more detailed description of which features are and are not
supported.

## Further Reading

While every publicly defined item in this crate should have some documentation attached to it,
most of the descriptions are quite terse for now and to the uninitiated will only make sense in
combination with the [MPI specification][MPIspec].

[MPIspec]: http://www.mpi-forum.org/docs/docs.html

License: MIT/Apache-2.0
