use crate::{data_types::*, ThemeConfig};
use serde::{Deserialize, Serialize};
use std::i64;
use tui::{
    style::{Color, Style},
    text::Span,
    widgets::{Block, Borders, ListState, TableState},
};

pub fn color_from_hex(hex: &str) -> Color {
    let hex = i64::from_str_radix(hex.strip_prefix("#").expect("Invalid color"), 16).unwrap();

    let r = ((hex & 0xFF0000) >> 16) as u8;
    let g = ((hex & 0x00FF00) >> 8) as u8;
    let b = (hex & 0x0000FF) as u8;

    Color::Rgb(r, g, b)
}

pub fn theme_color(theme: &ThemeConfig, style: &AppStyle) -> Option<Color> {
    theme.get(&style).map(|hex| color_from_hex(hex))
}

#[derive(Serialize, Deserialize, Debug, Clone, Hash, PartialEq, Eq)]
pub enum AppStyle {
    Border,
    FrameTitle,
    TabTitle,
    ContentText,
    Player,
}

fn fg_style(theme: &ThemeConfig, style: &AppStyle) -> Style {
    Style::default().fg(theme_color(&theme, &style).unwrap_or(Color::White))
}

pub fn get_style(theme: &ThemeConfig, style: &AppStyle) -> Style {
    fg_style(theme, &style)
}

pub fn frame_title_span(theme: &ThemeConfig, title: String) -> Span {
    return Span::styled(title, get_style(theme, &AppStyle::FrameTitle));
}

pub fn create_content_frame(theme: &ThemeConfig, title: String) -> Block {
    Block::default()
        .title(frame_title_span(&theme, title))
        .borders(Borders::ALL)
        .border_style(get_style(&theme, &AppStyle::Border))
}

pub fn table_style(theme: &ThemeConfig) -> Style {
    get_style(&theme, &AppStyle::ContentText)
}

#[derive(Debug, Clone)]
pub struct TabsState {
    pub titles: Vec<String>,
    pub index: usize,
}

impl TabsState {
    pub fn new(titles: Vec<&str>) -> TabsState {
        TabsState {
            titles: titles.iter().map(|t| (*t).into()).collect(),
            index: 0,
        }
    }

    pub fn next(&mut self) {
        self.index = (self.index + 1) % self.titles.len();
    }

    pub fn previous(&mut self) {
        if self.index > 0 {
            self.index -= 1;
        } else {
            self.index = self.titles.len() - 1;
        }
    }

    pub fn set(&mut self, index: usize) {
        if index < self.titles.len() {
            self.index = index
        }
    }
}

#[derive(Debug, Clone)]
pub struct StatefulList<T> {
    pub state: ListState,
    pub items: Vec<T>,
}

impl<T> StatefulList<T> {
    pub fn new() -> StatefulList<T> {
        StatefulList {
            state: ListState::default(),
            items: Vec::new(),
        }
    }

    pub fn with_items(items: Vec<T>) -> StatefulList<T> {
        StatefulList {
            state: ListState::default(),
            items,
        }
    }

    pub fn next(&mut self) {
        let i = match self.state.selected() {
            Some(i) => {
                if i >= self.items.len() - 1 {
                    0
                } else {
                    i + 1
                }
            }
            None => 0,
        };
        self.state.select(Some(i));
    }

    pub fn previous(&mut self) {
        let i = match self.state.selected() {
            Some(i) => {
                if i == 0 {
                    self.items.len() - 1
                } else {
                    i - 1
                }
            }
            None => 0,
        };
        self.state.select(Some(i));
    }

    pub fn unselect(&mut self) {
        self.state.select(None);
    }
}

#[derive(Debug, Clone)]
pub struct StatefulTable<T> {
    pub state: TableState,
    pub items: Vec<T>,
}

impl<T> Default for StatefulTable<T> {
    fn default() -> Self {
        Self::new()
    }
}

impl<T> StatefulTable<T> {
    pub fn new() -> Self {
        Self {
            state: TableState::default(),
            items: Vec::new(),
        }
    }

    pub fn with_items(items: Vec<T>) -> Self {
        Self {
            state: TableState::default(),
            items,
        }
    }

    pub fn index_from_selector(&self, selector: &ListEntrySelector) -> usize {
        use ListEntrySelector::*;

        match selector {
            Next => self
                .state
                .selected()
                .map(|s| (s as isize + 1).rem_euclid(self.items.len() as isize))
                .unwrap_or(0) as usize,
            Prev => self
                .state
                .selected()
                .map(|s| (s as isize - 1).rem_euclid(self.items.len() as isize))
                .unwrap_or(0) as usize,
            First => 0,
            Last => self.items.len() - 1,
            Selected => self.state.selected().unwrap_or(0),
            Nth(n) => n.rem_euclid(self.items.len()),
        }
    }

    pub fn item_from_selector<'a>(&'a self, selector: &ListEntrySelector) -> Option<&'a T> {
        self.items.get(self.index_from_selector(selector))
    }

    pub fn select(&mut self, selector: &ListEntrySelector) {
        if !self.items.is_empty() {
            self.state.select(Some(self.index_from_selector(selector)));
        } else {
            self.state.select(None);
        }
    }

    pub fn first(&mut self) {
        self.select(&ListEntrySelector::First)
    }

    pub fn last(&mut self) {
        self.select(&ListEntrySelector::Last)
    }

    pub fn next(&mut self) {
        self.select(&ListEntrySelector::Next)
    }

    pub fn previous(&mut self) {
        self.select(&ListEntrySelector::Prev)
    }
}
