use crate::{
    data_types::*,
    model::{library, playlists, progress_bar, queue, volume, Model},
    Config, MPDConnection, View,
};
use std::{
    error::Error,
    sync::{
        atomic::{AtomicBool, Ordering},
        mpsc, Condvar, Mutex, MutexGuard,
    },
};
use termion::{input::MouseTerminal, raw::IntoRawMode, screen::AlternateScreen};
use tui::{backend::TermionBackend, Terminal};

/**
 * Rules for aquiring a lock on connection and/or model to prevent deadlocks:
 * - always lock connection before locking model
 * - if a lock on model has be aqiired, release the lock before locking connection
 * Or simply: never keep a lock on model
 */
pub struct App {
    model: Mutex<Model>,
    model_cvar: Condvar,
    connection: Mutex<MPDConnection>,
    config: Config,
    view: Mutex<View>,
    quit: AtomicBool,
    rx: Mutex<mpsc::Receiver<Message>>,
    tx: Mutex<mpsc::Sender<Message>>,
}

impl Default for App {
    fn default() -> Self {
        Self::new()
    }
}

impl App {
    pub fn new() -> Self {
        let config = Config::new();
        let (tx, rx) = mpsc::channel();

        Self {
            model: Mutex::new(Model::new(&config)),
            model_cvar: Condvar::new(),
            connection: Mutex::new(MPDConnection::new(&config)),
            view: Mutex::new(View::new()),
            config,
            quit: AtomicBool::new(false),
            rx: Mutex::new(rx),
            tx: Mutex::new(tx),
        }
    }

    pub fn render_loop(&self) -> Result<(), Box<dyn Error>> {
        let mut terminal = {
            let stdout = std::io::stdout().into_raw_mode()?;
            let stdout = MouseTerminal::from(stdout);
            let stdout = AlternateScreen::from(stdout);
            let backend = TermionBackend::new(stdout);
            Terminal::new(backend)?
        };

        let mut model = self.model();
        self.view()
            .view(&mut terminal, &mut model, &self.config().theme)?;

        loop {
            model = self.wait(model);

            if self.has_quit() {
                break;
            }

            self.view()
                .view(&mut terminal, &mut model, &self.config().theme)?;
        }

        Ok(())
    }

    pub fn send_message(&self, message: Message) {
        let _ = self.tx.lock().unwrap().send(message);
    }

    pub fn rx(&self) -> MutexGuard<mpsc::Receiver<Message>> {
        self.rx.lock().unwrap()
    }

    pub fn has_quit(&self) -> bool {
        self.quit.load(Ordering::Relaxed)
    }

    pub fn quit(&self) {
        self.quit.store(true, Ordering::Relaxed);
    }

    pub fn config(&self) -> &Config {
        &self.config
    }

    pub fn connection(&self) -> MutexGuard<MPDConnection> {
        self.connection.lock().unwrap()
    }

    pub fn model(&self) -> MutexGuard<Model> {
        self.model.lock().unwrap()
    }

    pub fn view(&self) -> MutexGuard<View> {
        self.view.lock().unwrap()
    }

    pub fn wait<'m>(&self, model: MutexGuard<'m, Model>) -> MutexGuard<'m, Model> {
        self.model_cvar.wait(model).unwrap()
    }

    pub fn update_view(&self) {
        self.model_cvar.notify_one()
    }

    pub fn start_loading(&self) {
        self.model().loading += 1;
        self.update_view();
    }

    pub fn stop_loading(&self) {
        self.model().loading -= 1;
        self.update_view();
    }

    pub fn seek(&self, offset: i32) {
        let mut connection = self.connection();
        if let Ok(status) = connection.status() {
            if let Some(elapsed) = status.elapsed {
                let new_pos = elapsed.num_seconds() + (offset as i64);
                let _ = connection.rewind(new_pos);
            }
        }
    }

    pub fn handle_message(&self, message: Message) -> Result<(), Box<dyn Error>> {
        log!("Handling message: {:?}", &message);
        self.start_loading();

        use Message::*;
        match &message {
            UpdatePlayer => progress_bar::update(self),
            UpdateLibrary => library::update(self),
            UpdateQueue => queue::update(self),
            UpdatePlaylists => playlists::update(self),
            UpdateVolume => volume::update(self),

            Rescan => { self.connection().rescan()?; },
            Seek(offset) => self.seek(*offset),
            TogglePause => self.connection().toggle_pause()?,
            NextTab => self.model().tabs.next(),
            PrevTab => self.model().tabs.previous(),
            SetTab(tab) => self.model().tabs.set(tab.index()),
            ChangeVolume(v) => volume::change_volume(self, *v as i8),
            SetVolume(v) => volume::set_volume(self, *v as i8),
            Quit => self.quit(),
            _ => {}
        }

        let current_content = self.model().current_content();

        match current_content {
            TabContent::Library => self.handle_library_message(&message),
            TabContent::Queue => self.handle_queue_message(&message),
            TabContent::Playlists => self.handle_playlists_message(&message),
        }

        self.stop_loading();

        Ok(())
    }

    fn handle_library_message(&self, message: &Message) {
        use Message::*;
        match message {
            MoveDown => self.model().library.items.next(),
            MoveUp => self.model().library.items.previous(),
            MoveFirst => self.model().library.items.first(),
            MoveLast => self.model().library.items.last(),
            Play => library::play(self),
            ListExpand => library::state_down(self),
            ListCollapse => library::state_up(self),
            _ => {}
        }
    }

    fn handle_queue_message(&self, message: &Message) {
        use Message::*;
        match message {
            MoveDown => self.model().queue.items.next(),
            MoveUp => self.model().queue.items.previous(),
            MoveFirst => self.model().queue.items.first(),
            MoveLast => self.model().queue.items.last(),
            Play => queue::play(self),
            _ => {}
        }
    }

    fn handle_playlists_message(&self, message: &Message) {
        use Message::*;
        match message {
            MoveDown => self.model().playlists.items.next(),
            MoveUp => self.model().playlists.items.previous(),
            MoveFirst => self.model().playlists.items.first(),
            MoveLast => self.model().playlists.items.last(),
            Play => playlists::play(self),
            _ => {}
        }
    }
}
