use crate::ThemeConfig;
use std::i64;
use tui::{
    style::{Color, Style},
    text::Span,
    widgets::{Block, Borders, ListState, TableState},
};

pub fn color_from_hex(hex: &str) -> Color {
    let hex = i64::from_str_radix(hex.strip_prefix("#").expect("Invalid color"), 16).unwrap();

    let r = ((hex & 0xFF0000) >> 16) as u8;
    let g = ((hex & 0x00FF00) >> 8) as u8;
    let b = (hex & 0x0000FF) as u8;

    Color::Rgb(r, g, b)
}

pub fn theme_color(theme: &ThemeConfig, color: &str) -> Option<Color> {
    theme.get(color).map(|hex| color_from_hex(hex))
}

pub enum AppStyle {
    Border,
    FrameTitle,
    TabTitle,
    ContentText,
    Player,
}

fn fg_style(theme: &ThemeConfig, color: &str) -> Style {
    Style::default().fg(theme_color(&theme, color).unwrap_or(Color::White))
}

pub fn get_style(theme: &ThemeConfig, style: AppStyle) -> Style {
    use AppStyle::*;
    match style {
        Border => fg_style(&theme, "color_border"),
        FrameTitle => fg_style(&theme, "color_frame_title"),
        TabTitle => fg_style(&theme, "color_tab_title"),
        ContentText => fg_style(&theme, "color_text"),
        Player => fg_style(&theme, "color_player"),
    }
}

pub fn frame_title_span(theme: &ThemeConfig, title: String) -> Span {
    return Span::styled(title, get_style(theme, AppStyle::FrameTitle));
}

pub fn create_content_frame(theme: &ThemeConfig, title: String) -> Block {
    Block::default()
        .title(frame_title_span(&theme, title))
        .borders(Borders::ALL)
        .border_style(get_style(&theme, AppStyle::Border))
}

pub fn table_style(theme: &ThemeConfig) -> Style {
    get_style(&theme, AppStyle::ContentText)
}

#[derive(Debug, Clone)]
pub struct TabsState {
    pub titles: Vec<String>,
    pub index: usize,
}

impl TabsState {
    pub fn new(titles: Vec<&str>) -> TabsState {
        TabsState {
            titles: titles.iter().map(|t| (*t).into()).collect(),
            index: 0,
        }
    }

    pub fn next(&mut self) {
        self.index = (self.index + 1) % self.titles.len();
    }

    pub fn previous(&mut self) {
        if self.index > 0 {
            self.index -= 1;
        } else {
            self.index = self.titles.len() - 1;
        }
    }

    pub fn set(&mut self, index: usize) {
        if index < self.titles.len() {
            self.index = index
        }
    }
}

#[derive(Debug, Clone)]
pub struct StatefulList<T> {
    pub state: ListState,
    pub items: Vec<T>,
}

impl<T> StatefulList<T> {
    pub fn new() -> StatefulList<T> {
        StatefulList {
            state: ListState::default(),
            items: Vec::new(),
        }
    }

    pub fn with_items(items: Vec<T>) -> StatefulList<T> {
        StatefulList {
            state: ListState::default(),
            items,
        }
    }

    pub fn next(&mut self) {
        let i = match self.state.selected() {
            Some(i) => {
                if i >= self.items.len() - 1 {
                    0
                } else {
                    i + 1
                }
            }
            None => 0,
        };
        self.state.select(Some(i));
    }

    pub fn previous(&mut self) {
        let i = match self.state.selected() {
            Some(i) => {
                if i == 0 {
                    self.items.len() - 1
                } else {
                    i - 1
                }
            }
            None => 0,
        };
        self.state.select(Some(i));
    }

    pub fn unselect(&mut self) {
        self.state.select(None);
    }
}

#[derive(Debug, Clone)]
pub struct StatefulTable<T> {
    pub state: TableState,
    pub items: Vec<T>,
}

impl<T> Default for StatefulTable<T> {
    fn default() -> Self {
        Self::new()
    }
}

impl<T> StatefulTable<T> {
    pub fn new() -> Self {
        Self {
            state: TableState::default(),
            items: Vec::new(),
        }
    }

    pub fn with_items(items: Vec<T>) -> Self {
        Self {
            state: TableState::default(),
            items,
        }
    }

    pub fn first(&mut self) {
        if !self.items.is_empty() {
            self.state.select(Some(0));
        } else {
            self.state.select(None);
        }
    }

    pub fn last(&mut self) {
        if !self.items.is_empty() {
            self.state.select(Some(self.items.len() - 1));
        } else {
            self.state.select(None);
        }
    }

    pub fn next(&mut self) {
        if !self.items.is_empty() {
            let i = match self.state.selected() {
                Some(i) => {
                    if i >= self.items.len() - 1 {
                        0
                    } else {
                        i + 1
                    }
                }
                None => 0,
            };
            self.state.select(Some(i));
        } else {
            self.state.select(None);
        }
    }

    pub fn previous(&mut self) {
        if !self.items.is_empty() {
            let i = match self.state.selected() {
                Some(i) => {
                    if i == 0 {
                        self.items.len() - 1
                    } else {
                        i - 1
                    }
                }
                None => 0,
            };
            self.state.select(Some(i));
        } else {
            self.state.select(None);
        }
    }
}
