pub mod library;
pub mod playlists;
pub mod progress_bar;
pub mod queue;
pub mod volume;

use serde::{Deserialize, Serialize};
use std::sync::{Arc, Condvar, Mutex, MutexGuard};
use strum::IntoEnumIterator;
use strum_macros::{EnumIter, EnumString, IntoStaticStr};

use crate::{data_types::*, util::ui_util::TabsState, Config, Keys};

pub use library::LibraryModel;
pub use playlists::PlaylistsModel;
pub use progress_bar::ProgressBarModel;
pub use queue::QueueModel;
pub use volume::VolumeModel;

#[derive(
    Debug, Clone, EnumIter, EnumString, IntoStaticStr, PartialEq, Eq, Serialize, Deserialize,
)]
pub enum TabContent {
    Library,
    Playlists,
    Queue,
}

impl TabContent {
    pub fn input_context(&self) -> Vec<Context> {
        match self {
            TabContent::Library => vec![Context::Global, Context::Library, Context::List],
            TabContent::Playlists => vec![Context::Global, Context::Playlists, Context::List],
            TabContent::Queue => vec![Context::Global, Context::Queue, Context::List],
        }
    }

    pub fn index(&self) -> usize {
        match self {
            TabContent::Library => 0,
            TabContent::Playlists => 1,
            TabContent::Queue => 2,
        }
    }
}

#[derive(Debug, Clone)]
pub struct AppHandle(pub Arc<(Mutex<Model>, Condvar)>);

impl AppHandle {
    pub fn model(&self) -> MutexGuard<Model> {
        self.0 .0.lock().unwrap()
    }

    pub fn update_view(&self) {
        self.0 .1.notify_one()
    }

    pub fn start_loading(&self) {
        self.model().loading += 1;
        self.update_view();
    }

    pub fn stop_loading(&self) {
        self.model().loading -= 1;
        self.update_view();
    }
}

#[derive(Debug)]
pub struct Model {
    pub queue: QueueModel,
    pub library: LibraryModel,
    pub playlists: PlaylistsModel,

    pub progress_bar: ProgressBarModel,
    pub volume: VolumeModel,

    pub tabs: TabsState,

    pub keys: Keys,
    pub input_state: InputState,
    pub loading: u32,
}

impl Model {
    pub fn new(config: &Config) -> Self {
        Self {
            queue: QueueModel::new(),
            library: LibraryModel::new(),
            playlists: PlaylistsModel::new(),
            progress_bar: ProgressBarModel::new(),
            volume: VolumeModel::new(),
            tabs: create_tabs(),
            keys: Keys::new(&config),
            input_state: InputState::None,
            loading: 0,
        }
    }

    pub fn current_content(&self) -> TabContent {
        TabContent::iter()
            .nth(self.tabs.index)
            .expect("The index of tabs should never be outside the range of tabs.")
    }
}

fn create_tabs() -> TabsState {
    TabsState::new(TabContent::iter().map(|t| t.into()).collect())
}
