use crate::{model::QueueModel, AppBackend};

use mpd::Song;
use time::Duration;
use tui::{
    layout::{Constraint, Rect},
    style::{Color, Modifier, Style},
    terminal::Frame,
    widgets::{Block, Borders, Row, Table},
};

pub fn view(queue: &mut QueueModel, f: &mut Frame<AppBackend>, area: Rect) {
    let current_song_name: String = queue
        .current_song
        .clone()
        .map(|s| s.file)
        .unwrap_or_else(|| "".into());

    let items = queue.items.items.iter().map(|song| {
        let lines = parse_tags(&song).into_iter();
        if current_song_name == song.file {
            return Row::StyledData(lines, Style::default().add_modifier(Modifier::BOLD));
        }

        Row::Data(lines)
    });
    let items = Table::new(vec!["  #", "Artist", "Title", "🕔"].into_iter(), items)
        .block(
            Block::default()
                .title("Queue")
                .borders(Borders::ALL)
                .style(Style::default().fg(Color::Cyan)),
        )
        .highlight_style(Style::default().add_modifier(Modifier::BOLD))
        .highlight_symbol("→ ")
        .widths(&[
            Constraint::Percentage(5),
            Constraint::Percentage(40),
            Constraint::Percentage(40),
            Constraint::Percentage(15),
        ]);

    f.render_stateful_widget(items, area, &mut queue.items.state);
}

fn parse_tags(song: &Song) -> Vec<String> {
    let unknown_artist = String::from("Unknown Artist");
    vec![
        song.place.map(|p| p.pos).unwrap_or_else(|| 0).to_string(),
        song.tags
            .get("Artist")
            .unwrap_or(&unknown_artist)
            .clone(),
        song.title.clone().unwrap_or_else(|| song.file.clone()),
        format_duration(&song.duration.unwrap_or_else(|| Duration::seconds(0))),
    ]
}

fn format_duration(duration: &Duration) -> String {
    if duration.num_hours() > 0 {
        format!(
            "{:02}:{:02}:{:02}",
            duration.num_hours(),
            duration.num_minutes() % 60,
            duration.num_seconds() % 60
        )
    } else {
        format!(
            "{:02}:{:02}",
            duration.num_minutes() % 60,
            duration.num_seconds() % 60
        )
    }
}
