mod library;
mod playlists;
mod progress_bar;
mod queue;

use crate::{model::TabContent, AppBackend, Config, InputState, Model};

use std::error::Error;

use tui::{
    layout::{Constraint, Direction, Layout, Rect},
    style::{Color, Modifier, Style},
    terminal::Frame,
    text::{Span, Spans},
    widgets::{Block, Borders, Tabs},
    Terminal,
};

pub struct View {}

impl View {
    pub fn new() -> Self {
        Self {}
    }

    pub fn view(
        &mut self,
        terminal: &mut Terminal<AppBackend>,
        mut model: &mut Model,
        _config: &Config,
    ) -> Result<(), Box<dyn Error>> {
        terminal.draw(|f: &mut Frame<AppBackend>| {
            let size = f.size();
            let chunks = Layout::default()
                .direction(Direction::Vertical)
                .margin(0)
                .constraints(
                    [
                        Constraint::Length(3),
                        Constraint::Min(0),
                        Constraint::Length(3),
                    ]
                    .as_ref(),
                )
                .split(size);

            let block = Block::default().style(Style::default().fg(Color::Black));
            f.render_widget(block, size);

            self.render_tabs(&mut model, f, chunks[0]);

            match &model.current_content() {
                TabContent::Library => library::view(&mut model.library, f, chunks[1]),
                TabContent::Playlists => playlists::view(&mut model.playlists, f, chunks[1]),
                TabContent::Queue => queue::view(&mut model.queue, f, chunks[1]),
            }

            match &model.input_state {
                InputState::TextInput => {}
                InputState::None => {
                    progress_bar::view(&model.progress_bar, f, chunks[2]);
                }
            }
        })?;

        Ok(())
    }

    fn render_tabs(&mut self, model: &mut Model, f: &mut Frame<AppBackend>, area: Rect) {
        let mut titles = model
            .tabs
            .titles
            .iter()
            .map(|t| Spans::from(Span::styled(t.clone(), Style::default().fg(Color::Yellow))))
            .collect::<Vec<_>>();

        {
            let sequencer = &model.keys.sequencer;

            if !sequencer.sequence().is_empty() {
                titles.push(Spans::from(format!("{}", sequencer)));
            }
        }

        if model.loading > 0 {
            titles.push(Spans::from("◌"));
        }

        let tabs = Tabs::new(titles)
            .block(Block::default().borders(Borders::ALL))
            .select(model.tabs.index)
            .style(Style::default().fg(Color::Cyan))
            .highlight_style(Style::default().add_modifier(Modifier::BOLD));
        f.render_widget(tabs, area);
    }
}
