mod library;
mod playlists;
mod progress_bar;
mod queue;
mod volume;

use serde::{Deserialize, Serialize};
use std::{
    error::Error,
    sync::{Arc, Condvar, Mutex, MutexGuard},
};
use strum::IntoEnumIterator;
use strum_macros::{EnumIter, EnumString, IntoStaticStr};

use crate::{
    data_types::*,
    util::{ui_util::TabsState, MPDConnection},
    Config, Keys,
};

pub use library::LibraryModel;
pub use playlists::PlaylistsModel;
pub use progress_bar::ProgressBarModel;
pub use queue::QueueModel;
pub use volume::VolumeModel;

#[derive(
    Debug, Clone, EnumIter, EnumString, IntoStaticStr, PartialEq, Eq, Serialize, Deserialize,
)]
pub enum TabContent {
    Library,
    Playlists,
    Queue,
}

impl TabContent {
    pub fn input_context(&self) -> Vec<Context> {
        match self {
            TabContent::Library => vec![Context::Global, Context::Library, Context::List],
            TabContent::Playlists => vec![Context::Global, Context::Playlists, Context::List],
            TabContent::Queue => vec![Context::Global, Context::Queue, Context::List],
        }
    }

    pub fn index(&self) -> usize {
        match self {
            TabContent::Library => 0,
            TabContent::Playlists => 1,
            TabContent::Queue => 2,
        }
    }
}

#[derive(Debug, Clone)]
pub struct AppHandle(pub Arc<(Mutex<Model>, Condvar)>);

impl AppHandle {
    pub fn model(&self) -> MutexGuard<Model> {
        self.0 .0.lock().unwrap()
    }

    pub fn update_view(&self) {
        self.0 .1.notify_one()
    }

    pub fn start_loading(&self) {
        self.model().loading += 1;
        self.update_view();
    }

    pub fn stop_loading(&self) {
        self.model().loading -= 1;
        self.update_view();
    }
}

#[derive(Debug)]
pub struct Model {
    pub queue: QueueModel,
    pub library: LibraryModel,
    pub playlists: PlaylistsModel,

    pub progress_bar: ProgressBarModel,
    pub volume: VolumeModel,

    pub tabs: TabsState,

    pub keys: Keys,
    pub quit: bool,
    pub input_state: InputState,
    pub loading: u32,
}

impl Model {
    pub fn new(config: &Config) -> Self {
        Self {
            queue: QueueModel::new(),
            library: LibraryModel::new(),
            playlists: PlaylistsModel::new(),
            progress_bar: ProgressBarModel::new(),
            volume: VolumeModel::new(),
            tabs: create_tabs(),
            keys: Keys::new(&config),
            quit: false,
            input_state: InputState::None,
            loading: 0,
        }
    }

    pub fn current_content(&self) -> TabContent {
        TabContent::iter()
            .nth(self.tabs.index)
            .expect("The index of tabs should never be outside the range of tabs.")
    }
}

fn create_tabs() -> TabsState {
    TabsState::new(TabContent::iter().map(|t| t.into()).collect())
}

pub fn handle_message(
    app: &AppHandle,
    mut connection: &mut MPDConnection,
    message: Message,
) -> Result<(), Box<dyn Error>> {
    app.start_loading();

    use Message::*;
    match &message {
        UpdatePlayer => progress_bar::update(app, connection),
        UpdateLibrary => library::update(app, connection),
        UpdateQueue => queue::update(app, connection),
        UpdatePlaylists => playlists::update(app, connection),
        UpdateVolume => volume::update(app, connection),

        TogglePause => connection.toggle_pause()?,
        NextTab => app.model().tabs.next(),
        PrevTab => app.model().tabs.previous(),
        SetTab(tab) => app.model().tabs.set(tab.index()),
        VolumeUp => volume::change_volume(app, connection, 1),
        VolumeDown => volume::change_volume(app, connection, -1),
        Quit => app.model().quit = true,
        _ => {}
    }

    let current_content = app.model().current_content();

    match current_content {
        TabContent::Library => handle_library_message(&app, &mut connection, &message),
        TabContent::Queue => handle_queue_message(&app, &mut connection, &message),
        TabContent::Playlists => handle_playlists_message(&app, &mut connection, &message),
    }

    app.stop_loading();

    Ok(())
}

fn handle_library_message(app: &AppHandle, connection: &mut MPDConnection, message: &Message) {
    use Message::*;
    match message {
        MoveDown => app.model().library.items.next(),
        MoveUp => app.model().library.items.previous(),
        MoveFirst => app.model().library.items.first(),
        MoveLast => app.model().library.items.last(),
        Play => library::play(app, connection),
        ListExpand => library::state_down(app, connection),
        ListCollapse => library::state_up(app, connection),
        _ => {}
    }
}

fn handle_queue_message(app: &AppHandle, connection: &mut MPDConnection, message: &Message) {
    use Message::*;
    match message {
        MoveDown => app.model().queue.items.next(),
        MoveUp => app.model().queue.items.previous(),
        MoveFirst => app.model().queue.items.first(),
        MoveLast => app.model().queue.items.last(),
        Play => queue::play(app, connection),
        _ => {}
    }
}

fn handle_playlists_message(app: &AppHandle, connection: &mut MPDConnection, message: &Message) {
    use Message::*;
    match message {
        MoveDown => app.model().playlists.items.next(),
        MoveUp => app.model().playlists.items.previous(),
        MoveFirst => app.model().playlists.items.first(),
        MoveLast => app.model().playlists.items.last(),
        Play => playlists::play(app, connection),
        _ => {}
    }
}
