use crate::{model::AppHandle, util::ui_util::StatefulTable, MPDConnection};
use mpd::{Query, Song, Term};

#[derive(Debug, Clone)]
struct Selection(usize, String);

#[derive(Debug, Clone)]
enum State {
    Artists,
    Artist(Selection),
    Album(Selection, Selection),
    Song(Selection, Selection, Selection),
}

use State::*;

#[derive(Debug, Clone)]
pub struct LibraryModel {
    pub items: StatefulTable<String>,
    state: State,
}

impl Default for LibraryModel {
    fn default() -> Self {
        Self::new()
    }
}

impl LibraryModel {
    pub fn new() -> Self {
        Self {
            items: StatefulTable::new(),
            state: State::Artists,
        }
    }

    pub fn get_title(&self) -> String {
        match self.state.clone() {
            Artists => "Artists".into(),
            Artist(artist) => format!("Artists → {}", artist.1),
            Album(artist, album) => format!("Artists → {} → {}", artist.1, album.1),
            Song(artist, album, song) => {
                format!("Artists → {} → {} → {}", artist.1, album.1, song.1)
            }
        }
    }

    fn get_next_state(&self) -> Option<State> {
        if let Some(index) = self.items.state.selected() {
            if let Some(selected) = self.items.items.get(index) {
                let state = self.state.clone();
                return Some(match state {
                    Artists => Artist(Selection(index, selected.clone())),
                    Artist(artist) => Album(artist, Selection(index, selected.clone())),
                    Album(ar, al) => Song(ar, al, Selection(index, selected.clone())),
                    Song(ar, al, so) => Song(ar, al, so),
                });
            }
        }

        None
    }
}

pub fn play(app: &AppHandle, connection: &mut MPDConnection) {
    let state = app.model().library.get_next_state();

    if let Some(next_state) = state {
        let _ = connection.clear();
        queue_add(connection, next_state);
        let _ = connection.play();
    }
}

fn queue_add(connection: &mut MPDConnection, next_state: State) {
    match next_state {
        Artists => {
            unreachable!()
        }
        Artist(artist) => queue_artist(connection, &artist.1),
        Album(_, album) => queue_album(connection, &album.1),
        Song(_, album, song) => queue_song(connection, &album.1, &song.1),
    };
}

fn queue_song(connection: &mut MPDConnection, album: &str, song: &str) {
    let mut query = Query::new();
    let query = query
        .and(Term::Tag("Title".into()), song)
        .and(Term::Tag("Album".into()), album);
    let _ = connection.findadd(query);
}

fn queue_album(connection: &mut MPDConnection, album: &str) {
    let _ = connection.findadd(Query::new().and(Term::Tag("Album".into()), album));
}

fn queue_artist(connection: &mut MPDConnection, artist: &str) {
    let _ = connection
        .list(
            &Term::Tag("Album".into()),
            Query::new().and(Term::Tag("Artist".into()), artist),
        )
        .map(|albums| {
            albums
                .iter()
                .for_each(|album| queue_album(connection, album));
        });
}

fn list_all(app: &AppHandle, connection: &mut MPDConnection, term: &str) {
    let list = connection
        .list(&mpd::Term::Tag(term.into()), &Query::new())
        .unwrap_or_else(|_| Vec::new());

    app.model().library.items = StatefulTable::with_items(list);
}

fn list(
    app: &AppHandle,
    connection: &mut MPDConnection,
    term: &str,
    query_tag: &str,
    query_value: &str,
) {
    let mut query = Query::new();
    let query = query.and(mpd::Term::Tag(query_tag.into()), query_value);
    let list = connection
        .list(&mpd::Term::Tag(term.into()), &query)
        .unwrap_or_else(|_| Vec::new());

    app.model().library.items = StatefulTable::with_items(list);
}

fn find_song(
    connection: &mut MPDConnection,
    album_name: String,
    song_name: String,
) -> Option<Song> {
    let mut query = Query::new();
    let query = query
        .and(Term::Tag("Title".into()), song_name)
        .and(Term::Tag("Album".into()), album_name);
    connection
        .find(query, None)
        .ok()
        .map(|songs| songs.first().unwrap().clone())
}

pub fn update(app: &AppHandle, connection: &mut MPDConnection) {
    let state = app.model().library.state.clone();
    match state {
        Artists => list_all(&app, connection, "AlbumArtist"),
        Artist(artist) => list(&app, connection, "Album", "Artist", &artist.1),
        Album(_, album) => list(&app, connection, "Title", "Album", &album.1),
        Song(_, _, song) => list(&app, connection, "Title", "Title", &song.1),
    }
}

pub fn state_down(app: &AppHandle, connection: &mut MPDConnection) {
    let state = app.model().library.get_next_state();
    if let Some(state) = state {
        match state.clone() {
            Artists => list_all(&app, connection, "AlbumArtist"),
            Artist(artist) => list(&app, connection, "Album", "Artist", &artist.1),
            Album(_, album) => list(&app, connection, "Title", "Album", &album.1),
            Song(_, _, song) => list(&app, connection, "Title", "Title", &song.1),
        };

        app.model().library.state = state;
    }
}

pub fn state_up(app: &AppHandle, connection: &mut MPDConnection) {
    let mut new_index = None;

    let next_state = match app.model().library.state.clone() {
        Artists => Artists,
        Artist(artist) => {
            new_index = Some(artist.0);
            Artists
        }
        Album(artist, album) => {
            new_index = Some(album.0);
            Artist(artist)
        }
        Song(artist, album, song) => {
            new_index = Some(song.0);
            Album(artist, album)
        }
    };

    match next_state.clone() {
        Artists => list_all(&app, connection, "AlbumArtist"),
        Artist(artist) => list(&app, connection, "Album", "Artist", &artist.1),
        Album(_, album) => list(&app, connection, "Title", "Album", &album.1),
        Song(_, _, song) => list(&app, connection, "Title", "Title", &song.1),
    };

    let mut library = &mut app.model().library;
    library.state = next_state;
    library.items.state.select(new_index);
}
