#![allow(dead_code)]
#![allow(unreachable_code)]

#[macro_use]
mod util;
mod config;
mod data_types;
mod model;
mod view;

use crate::{
    data_types::*,
    model::*,
    util::{
        event::{Event, Events},
        keys::Keys,
        log::log_init,
    },
    view::View,
};
use backtrace::Backtrace;
use config::Config;
use mpd::{
    idle::{Idle, Subsystem},
    Channel,
};
use std::{error::Error, panic, panic::PanicInfo};
pub use util::MPDConnection;

use std::{
    sync::{mpsc, Arc, Condvar, Mutex},
    thread,
};

use termion::{
    input::MouseTerminal,
    raw::{IntoRawMode, RawTerminal},
    screen::AlternateScreen,
};
use tui::{backend::TermionBackend, Terminal};

const UPDATE_INTERVAL: u64 = 1000;
pub const NUM_WORKERS: usize = 5;

pub type AppBackend = TermionBackend<AlternateScreen<MouseTerminal<RawTerminal<std::io::Stdout>>>>;

pub fn get_subsytems() -> Vec<Subsystem> {
    vec![Subsystem::Message, Subsystem::Player]
}

pub fn control_thread(app: AppHandle, config: Arc<Config>, rx: mpsc::Receiver<Message>) {
    let rx = Arc::new(Mutex::new(rx));
    let mut workers = Vec::with_capacity(NUM_WORKERS);

    for id in 0..NUM_WORKERS {
        let rx = rx.clone();
        let config = config.clone();
        let app = app.clone();

        workers.push(thread::spawn(move || {
            let mut connection = MPDConnection::new(&config);

            loop {
                let lock = rx.lock().unwrap();
                if let Ok(message) = lock.recv() {
                    std::mem::drop(lock);
                    log!("Worker {} handling {:?}", id, message);
                    let _ = handle_message(&app, &mut connection, message.clone());
                    if message == Message::Quit {
                        break;
                    }
                }
            }
        }));
    }

    workers.into_iter().for_each(|w| {
        let _ = w.join();
    });
}

pub fn mpd_listen_thread(config: Arc<Config>, tx: mpsc::Sender<Message>) {
    let mut connection = MPDConnection::new(&config);

    // This channel is used to break out of "idle_guard.get()"
    connection
        .subscribe(Channel::new("mpc_rs_msg").unwrap())
        .unwrap();

    let subsystems = get_subsytems();

    let _ = tx.send(Message::UpdateLibrary);
    let _ = tx.send(Message::UpdateQueue);
    let _ = tx.send(Message::UpdatePlayer);
    let _ = tx.send(Message::UpdatePlaylists);
    let _ = tx.send(Message::UpdateVolume);

    loop {
        let idle_guard = connection.idle(&subsystems).unwrap();

        if let Ok(subsystems) = idle_guard.get() {
            // When recieving a message, we must check if the
            // main thread is trying to join this thread
            let messages = connection.readmessages().unwrap();

            let mut stop = false;
            for message in messages {
                if &message.message[..] == "stop" { stop = true }
            }

            // If so we break out of the loop
            if stop {
                break;
            }

            subsystems
                .into_iter()
                .map(|s| s.into())
                .for_each(|message: Message| {
                    if message == Message::Quit {
                        for _ in 0..NUM_WORKERS {
                            let _ = tx.send(message.clone());
                        }
                    } else {
                        let _ = tx.send(message);
                    }
                });
        }
    }
}

pub fn input_thread(app: AppHandle, tx: mpsc::Sender<Message>) {
    let events = Events::new();

    loop {
        let event = events.next().unwrap();

        if let Event::Input(input) = event {
            let mut model = app.model();
            let mut status = false;

            if let Some(message) = model
                .keys
                .get_message(&model.current_content().input_context(), input)
            {
                status = true;
                if message == Message::Quit {
                    for _ in 0..NUM_WORKERS {
                        let _ = tx.send(message.clone());
                    }
                } else {
                    let _ = tx.send(message);
                }
            }

            model.keys.set_status(status, input);
        }

        app.update_view();
        if app.model().quit {
            break;
        }
    }
}

pub fn render_thread(app: AppHandle, config: Arc<Config>) -> Result<(), Box<dyn Error>> {
    let mut view = View::new();

    let mut terminal = {
        let stdout = std::io::stdout().into_raw_mode()?;
        let stdout = MouseTerminal::from(stdout);
        let stdout = AlternateScreen::from(stdout);
        let backend = TermionBackend::new(stdout);
        Terminal::new(backend)?
    };

    let mut model = app.model();
    view.view(&mut terminal, &mut model, &config)?;

    loop {
        model = app.0 .1.wait(model).unwrap();

        if model.quit {
            break;
        }

        view.view(&mut terminal, &mut model, &config)?;
    }

    Ok(())
}

fn main() -> Result<(), Box<dyn Error>> {
    panic::set_hook(Box::new(panic_hook));
    log_init();

    // Load config file
    let config = Arc::new(Config::new());
    let app = AppHandle(Arc::new((Mutex::new(Model::new(&config)), Condvar::new())));

    let (tx, rx) = mpsc::channel();

    let mpd_listen = {
        let config = config.clone();
        let tx = tx.clone();

        thread::spawn(move || mpd_listen_thread(config, tx))
    };

    let control = {
        let app = app.clone();
        let config = config.clone();

        thread::spawn(move || control_thread(app, config, rx))
    };

    let input = {
        let app = app.clone();
        let tx = tx.clone();

        thread::spawn(move || input_thread(app, tx))
    };

    render_thread(app, config.clone())?;

    let mut connection = MPDConnection::new(&config);
    let _ = connection.sendmessage(Channel::new("mpc_rs_msg").unwrap(), "stop");
    let _ = tx.send(Message::Quit);
    mpd_listen.join().unwrap();
    control.join().unwrap();
    input.join().unwrap();

    Ok(())
}

fn panic_hook(info: &PanicInfo<'_>) {
    if cfg!(debug_assertions) {
        let location = info.location().unwrap();

        let msg = match info.payload().downcast_ref::<&'static str>() {
            Some(s) => *s,
            None => match info.payload().downcast_ref::<String>() {
                Some(s) => &s[..],
                None => "Box<Any>",
            },
        };

        let stacktrace: String = format!("{:?}", Backtrace::new()).replace('\n', "\n\r");

        println!(
            "{}thread '<unnamed>' panicked at '{}', {}\n\r{}",
            termion::screen::ToMainScreen,
            msg,
            location,
            stacktrace
        );
    }
}
