extern crate proc_macro;

use proc_macro::*;

/// Places required C interface code in you Mosaiq app.
#[proc_macro_attribute]
pub fn app(_attr: TokenStream, item: TokenStream) -> TokenStream {
    let input = syn::parse_macro_input!(item as syn::File);

    let items = input.items;

    let result = quote::quote! {
        use mosaiq_app_api::*;

        static mut __INTERCOM: *mut Intercom = std::ptr::null_mut();

        #[no_mangle]
        pub unsafe extern "C" fn mosaiq_app_on_initialization(
            communicator: *mut mosaiq_app_c_bindings::MosaiqIntercomAppBroker,
        ) {
            let intercom = Box::new(Intercom {
                communicator: communicator,
                publications: Default::default(),
            });
            __INTERCOM = Box::into_raw(intercom);
            on_start(&*__INTERCOM);
        }

        #[no_mangle]
        pub unsafe extern "C" fn mosaiq_app_on_cyclic_execution() {
            on_cycle(&*__INTERCOM)
        }

        #[no_mangle]
        pub unsafe extern "C" fn mosaiq_app_on_exit() {
            on_stop(&*__INTERCOM)
        }

        #(#items)*
    };

    result.into()
}

#[proc_macro_attribute]
pub fn test_suite(_attr: TokenStream, item: TokenStream) -> TokenStream {
    let input = syn::parse_macro_input!(item as syn::ItemMod);

    let name = &input.ident;
    let body = &input.content.unwrap().1;
    let attrs = &input.attrs;

    let result = quote::quote! {

        #(#attrs)*
        #[cfg(test)]
        mod #name {

            #![allow(non_upper_case_globals)]
            #![allow(non_camel_case_types)]
            #![allow(non_snake_case)]

            use super::mosaiq_app_c_bindings;
            use mosaiq_app_api::testing::{TestSystem, InspectIntercom};

            static mut SYSTEM: *mut TestSystem = std::ptr::null_mut();

            unsafe extern "C" fn write(
                self_: *mut mosaiq_app_c_bindings::MosaiqIntercomPublication,
                value: *const ::std::os::raw::c_void,
            ) -> mosaiq_app_c_bindings::MosaiqIntercomWriteResult {
                let len = *(value as *const u32) as usize;
                let v = std::slice::from_raw_parts(value as *const u8, len + 4).to_vec();
                (*SYSTEM).published_values.push(v);
                mosaiq_app_c_bindings::MosaiqIntercomWriteResult_MOSAIQ_WRITE_SUCCESS
            }

            unsafe extern "C" fn write_sized(
                self_: *mut mosaiq_app_c_bindings::MosaiqIntercomPublication,
                value: *const ::std::os::raw::c_void,
                size: ::std::os::raw::c_ulong,
            ) -> mosaiq_app_c_bindings::MosaiqIntercomWriteResult {
                let v = std::slice::from_raw_parts(value as *const u8, size as usize).to_vec();
                (*SYSTEM).published_values.push(v);
                mosaiq_app_c_bindings::MosaiqIntercomWriteResult_MOSAIQ_WRITE_SUCCESS
            }

            unsafe extern "C" fn publish(
                self_: *mut mosaiq_app_c_bindings::MosaiqIntercomAppBroker,
                port: *mut *mut mosaiq_app_c_bindings::MosaiqIntercomPublication,
                topic: *const ::std::os::raw::c_char,
                type_: mosaiq_app_c_bindings::MosaiqIntercomType,
                defaultValue: *const ::std::os::raw::c_void,
                defaultValueSize: ::std::os::raw::c_ulong,
            ) -> mosaiq_app_c_bindings::MosaiqIntercomPublishResult {

                let topic_string = std::ffi::CStr::from_ptr(topic)
                    .to_str()
                    .unwrap()
                    .to_string();

                (*SYSTEM).publications.push(mosaiq_app_c_bindings::MosaiqIntercomPublication {
                    write: Some(write),
                    write_sized: Some(write_sized),
                    m_topic: std::ptr::null_mut(),
                });

                *port = (*SYSTEM).publications.last_mut().unwrap() as *mut mosaiq_app_c_bindings::MosaiqIntercomPublication;

                mosaiq_app_c_bindings::MosaiqIntercomPublishResult_MOSAIQ_PUBLISH_SUCCESS
            }

            unsafe extern "C" fn subscribe(
                self_: *mut mosaiq_app_c_bindings::MosaiqIntercomAppBroker,
                port: *mut *mut mosaiq_app_c_bindings::MosaiqIntercomSubscription,
                topic: *const ::std::os::raw::c_char,
                type_: mosaiq_app_c_bindings::MosaiqIntercomType,
            ) -> mosaiq_app_c_bindings::MosaiqIntercomSubscribeResult {
                mosaiq_app_c_bindings::MosaiqIntercomSubscribeResult_MOSAIQ_SUBSCRIBE_SUCCESS
            }



            #(#body)*
        }
    };

    result.into()
}

#[proc_macro_attribute]
pub fn test(_attr: TokenStream, item: TokenStream) -> TokenStream {
    let input = syn::parse_macro_input!(item as syn::ItemFn);

    let ret = &input.sig.output;
    let name = &input.sig.ident;
    let inputs = &input.sig.inputs;
    let body = &input.block;
    let attrs = &input.attrs;

    let result = quote::quote! {

        #[test]
        #(#attrs)*
        fn #name() #ret {

            let communicator = Box::new(mosaiq_app_c_bindings::MosaiqIntercomAppBroker {
                version: 0,
                m_impl: std::ptr::null_mut(),
                publish: Some(publish),
                subscribe: Some(subscribe),
            });

            let communicator = Box::into_raw(communicator);

            let intercom = Intercom {
                communicator,
                publications: Default::default()
            };

            fn inner_test(#inputs) {
                #body
            }

            unsafe {
                SYSTEM = Box::into_raw(Box::new(TestSystem {
                    publications: Vec::new(),
                    published_values: Vec::new()
                }));

                inner_test(&intercom, &(*SYSTEM))
            }
        }
    };

    result.into()
}
