#ifndef MOSAIQ_SDK_ERROR_H
#define MOSAIQ_SDK_ERROR_H

typedef enum MosaiqIntercomPublishResult
{
  MOSAIQ_PUBLISH_SUCCESS,
  /// This error occurs when another app already published this topic
  MOSAIQ_PUBLISH_ALREADY_PUBLISHED,
  /// This error occurs when a subscriber already subscribed to this topic using another type.
  MOSAIQ_PUBLISH_SUBSCRIBER_TYPE_MISMATCH,
  /// You may not publish topics during cyclic execution or teardown.
  MOSAIQ_PUBLISH_FORBIDDEN_OUTSIDE_INITIALIZATION
} MosaiqIntercomPublishResult;

typedef enum MosaiqIntercomSubscribeResult
{
  MOSAIQ_SUBSCRIBE_SUCCESS,
  /// This may happen when the subscriber subscribes before the publisher publishes. Which is not
  /// an error.
  MOSAIQ_SUBSCRIBE_SUCCESS_NO_PUBLISHER,
  /// Cannot subscribe to a topic by a different type than the pulisher published.
  MOSAIQ_SUBSCRIBE_TYPE_MISMATCH,
  /// You may not subscribe to topics during cyclic execution or teardown.
  MOSAIQ_SUBSCRIBE_FORBIDDEN_OUTSIDE_INITIALIZATION
} MosaiqIntercomSubscribeResult;

typedef enum MosaiqIntercomWriteResult
{
  MOSAIQ_WRITE_SUCCESS,
  /// This indicates a user error. Only use ports returned by a previous successful publish.
  MOSAIQ_WRITE_INVALID_PORT,
  /// A write with variable size cannot exceed a certain limit.
  MOSAIQ_WRITE_TOO_LARGE,
  /// A string and other types with variable size cannot be written without size information.
  MOSAIQ_WRITE_MISSING_SIZE_INFORMATION,
  /// Writing a variable of fixed known size with a different size is not allowed.
  MOSAIQ_WRITE_UNEXPECTED_SIZE
} MosaiqIntercomWriteResult;

typedef enum MosaiqIntercomReadResult
{
  MOSAIQ_READ_SUCCESS,
  /// Reading using the MosaiqIntercomReader is still necessary. Read has not been done yet.
  MOSAIQ_READ_RESULT_DEFERRED,
  /// This indicates a user error. Only use ports returned by a previous successful subscribe.
  MOSAIQ_READ_INVALID_PORT,
  /// A string and other types with variable size cannot be read without a provided buffer of given
  /// size.
  MOSAIQ_READ_MISSING_SIZE_INFORMATION,
  /// Reading a variable of fixed known size with a different size is not allowed. This is also
  /// reported when you try to read more bytes than there are available.
  MOSAIQ_READ_UNEXPECTED_SIZE,
  /// You cannot read fixed size arithmetic types with an offset. This is not allowed.
  MOSAIQ_READ_OFFSET_NOT_ALLOWED,
  /// Dont use the read_fixed function for arithmetic types, use read instead.
  MOSAIQ_READ_USE_REGULAR_READ,
  /// The topic was not published yet and reading results in default value of the constructed type
  MOSAIQ_READ_UNPUBLISHED_VALUE
} MosaiqIntercomReadResult;

static inline char const* mosaiq_describe_publish_result(MosaiqIntercomPublishResult result)
{
  switch (result)
  {
    case (MOSAIQ_PUBLISH_SUCCESS):
    {
      return "Success!";
    }
    case (MOSAIQ_PUBLISH_ALREADY_PUBLISHED):
    {
      return "Another app already published this topic.";
    }
    case (MOSAIQ_PUBLISH_SUBSCRIBER_TYPE_MISMATCH):
    {
      return "Another app subscribed to this topic with a different type.";
    }
    case (MOSAIQ_PUBLISH_FORBIDDEN_OUTSIDE_INITIALIZATION):
    {
      return "Publishing is only possible from within the on_initialization function.";
    }
  }
  return "Unknown result.";
}

static inline char const* mosaiq_describe_subscribe_result(MosaiqIntercomSubscribeResult result)
{
  switch (result)
  {
    case (MOSAIQ_SUBSCRIBE_SUCCESS):
    {
      return "Success!";
    }
    case (MOSAIQ_SUBSCRIBE_SUCCESS_NO_PUBLISHER):
    {
      return "No app published this topic yet, but can still happen.";
    }
    case (MOSAIQ_SUBSCRIBE_TYPE_MISMATCH):
    {
      return "Another app published this topic with a different type.";
    }
    case (MOSAIQ_SUBSCRIBE_FORBIDDEN_OUTSIDE_INITIALIZATION):
    {
      return "Subscribing is only possible from within the on_initialization function.";
    }
  }
  return "Unknown result.";
}

static inline char const* mosaiq_describe_write_result(MosaiqIntercomWriteResult result)
{
  switch (result)
  {
    case (MOSAIQ_WRITE_SUCCESS):
    {
      return "Success!";
    }
    case (MOSAIQ_WRITE_INVALID_PORT):
    {
      return "This indicates a user error. Only use ports returned by a previous successful "
             "publish.";
    }
    case (MOSAIQ_WRITE_TOO_LARGE):
    {
      return "A write with variable size cannot exceed a certain limit.";
    }
    case (MOSAIQ_WRITE_MISSING_SIZE_INFORMATION):
    {
      return "A string and other types with variable size cannot be written without size "
             "information.";
    }
    case (MOSAIQ_WRITE_UNEXPECTED_SIZE):
    {
      return "Writing a variable of fixed known size with a different size is not allowed.";
    }
  }
  return "Unknown result.";
}

static inline char const* mosaiq_describe_read_result(MosaiqIntercomReadResult result)
{
  switch (result)
  {
    case (MOSAIQ_READ_SUCCESS):
    {
      return "Success!";
    }
    case (MOSAIQ_READ_INVALID_PORT):
    {
      return "This indicates a user error. Only use ports returned by a previous successful "
             "subscribe.";
    }
    case (MOSAIQ_READ_MISSING_SIZE_INFORMATION):
    {
      return "A string and other types with variable size cannot be read without a provided buffer "
             "of given size.";
    }
    case (MOSAIQ_READ_UNEXPECTED_SIZE):
    {
      return "Passed size is too large or not correct for fixed size variable.";
    }
    case (MOSAIQ_READ_RESULT_DEFERRED):
    {
      return "The read operation is not yet complete. Please continue reading using the reader.";
    }
    case (MOSAIQ_READ_OFFSET_NOT_ALLOWED):
    {
      return "Fixed size arithmetic types cannot be read with a non-zero offset.";
    }
    case (MOSAIQ_READ_USE_REGULAR_READ):
    {
      return "Use the regular read function for arithmetic types.";
    }
    case (MOSAIQ_READ_UNPUBLISHED_VALUE):
    {
      return "The topic was not published yet. The value is default constructed from given type.";
    }
  }
  return "Unknown result.";
}

static inline int mosaiq_publish_was_successful(MosaiqIntercomPublishResult result)
{
  // NOLINTNEXTLINE(readability-implicit-bool-conversion)
  return result == MOSAIQ_PUBLISH_SUCCESS;
}

static inline int mosaiq_read_was_successful(MosaiqIntercomReadResult result)
{
  // NOLINTNEXTLINE(readability-implicit-bool-conversion)
  return result == MOSAIQ_READ_SUCCESS || result == MOSAIQ_READ_RESULT_DEFERRED;
}

static inline int mosaiq_write_was_successful(MosaiqIntercomWriteResult result)
{
  // NOLINTNEXTLINE(readability-implicit-bool-conversion)
  return result == MOSAIQ_WRITE_SUCCESS;
}

static inline int mosaiq_subscribe_was_successful(MosaiqIntercomSubscribeResult result)
{
  // NOLINTNEXTLINE(readability-implicit-bool-conversion)
  return result == MOSAIQ_SUBSCRIBE_SUCCESS || result == MOSAIQ_SUBSCRIBE_SUCCESS_NO_PUBLISHER;
}

#endif
