use std::{cell::UnsafeCell, collections::HashMap, fmt::Debug};

pub use libc;
pub use mosaiq_app_c_bindings;
pub use mosaiq_app_macros::app;

/// A computed value that can be published to the Mosaiq Intercom.
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub enum Value {
    U8(u8),
    U16(u16),
    U32(u32),
    U64(u64),
}

impl From<u8> for Value {
    fn from(value: u8) -> Self {
        Value::U8(value)
    }
}

impl From<u16> for Value {
    fn from(value: u16) -> Self {
        Value::U16(value)
    }
}

impl From<u32> for Value {
    fn from(value: u32) -> Self {
        Value::U32(value)
    }
}

impl From<u64> for Value {
    fn from(value: u64) -> Self {
        Value::U64(value)
    }
}

impl Value {
    /// Returns the value as u8 if it is of that type or None.
    pub fn as_u8(&self) -> Option<&u8> {
        match self {
            Value::U8(v) => Some(v),
            _ => None,
        }
    }

    /// Returns the value as u16 if it is of that type or None.
    pub fn as_u16(&self) -> Option<&u16> {
        match self {
            Value::U16(v) => Some(v),
            _ => None,
        }
    }

    /// Returns the value as u32 if it is of that type or None.
    pub fn as_u32(&self) -> Option<&u32> {
        match self {
            Value::U32(v) => Some(v),
            _ => None,
        }
    }

    /// Returns the value as u64 if it is of that type or None.
    pub fn as_u64(&self) -> Option<&u64> {
        match self {
            Value::U64(v) => Some(v),
            _ => None,
        }
    }
}

/// The Intercom allows to register topics and send computer Values to these topics so other Mosaiq apps can read value streams.
pub struct Intercom {
    pub communicator: *mut mosaiq_app_c_bindings::MosaiqIntercomAppBroker,
    pub publications:
        UnsafeCell<HashMap<&'static str, *mut mosaiq_app_c_bindings::MosaiqIntercomPublication>>,
}

impl Intercom {
    /// Create a topic for a given name and default value.
    /// The type of the default value determines the value type of the topic.
    pub fn create_topic(&self, topic: &'static str, default_value: Value) {
        unsafe {
            let communicator = *self.communicator;

            let mut publication = std::ptr::null_mut();

            let publish = communicator.publish.unwrap();

            match default_value {
                Value::U8(v) => {
                    publish(
                        //self
                        self.communicator,
                        //publish
                        &mut publication
                            as *mut *mut mosaiq_app_c_bindings::MosaiqIntercomPublication,
                        // topic
                        topic.as_ptr() as *const libc::c_char,
                        // type
                        mosaiq_app_c_bindings::MosaiqIntercomType_MOSAIQ_INTERCOM_INT8,
                        // default value
                        v as *const _,
                        // default value size
                        1,
                    );
                }
                Value::U16(v) => {
                    publish(
                        //self
                        self.communicator,
                        //publish
                        &mut publication
                            as *mut *mut mosaiq_app_c_bindings::MosaiqIntercomPublication,
                        // topic
                        topic.as_ptr() as *const libc::c_char,
                        // type
                        mosaiq_app_c_bindings::MosaiqIntercomType_MOSAIQ_INTERCOM_INT16,
                        // default value
                        v as *const _,
                        // default value size
                        1,
                    );
                }
                Value::U32(v) => {
                    publish(
                        //self
                        self.communicator,
                        //publish
                        &mut publication
                            as *mut *mut mosaiq_app_c_bindings::MosaiqIntercomPublication,
                        // topic
                        topic.as_ptr() as *const libc::c_char,
                        // type
                        mosaiq_app_c_bindings::MosaiqIntercomType_MOSAIQ_INTERCOM_INT32,
                        // default value
                        v as *const _,
                        // default value size
                        1,
                    );
                }
                Value::U64(v) => {
                    publish(
                        //self
                        self.communicator,
                        //publish
                        &mut publication
                            as *mut *mut mosaiq_app_c_bindings::MosaiqIntercomPublication,
                        // topic
                        topic.as_ptr() as *const libc::c_char,
                        // type
                        mosaiq_app_c_bindings::MosaiqIntercomType_MOSAIQ_INTERCOM_INT64,
                        // default value
                        v as *const _,
                        // default value size
                        1,
                    );
                }
            }
            (*self.publications.get()).insert(topic, publication);
        }
    }

    /// Publish the given value to the specified topic.
    pub fn publish(&self, topic: &'static str, value: Value) {
        unsafe {
            match (*self.publications.get()).get(topic).cloned() {
                Some(publication) => {
                    let _write = (*publication).write.unwrap();
                    let write_sized = (*publication).write_sized.unwrap();
                    match value {
                        Value::U8(v) => {
                            let p = &v as *const u8;
                            write_sized(publication, p as *const libc::c_void, 1);
                        }
                        Value::U16(v) => {
                            let p = &v as *const u16;
                            write_sized(publication, p as *const libc::c_void, 2);
                        }
                        Value::U32(v) => {
                            let p = &v as *const u32;
                            write_sized(publication, p as *const libc::c_void, 4);
                        }
                        Value::U64(v) => {
                            let p = &v as *const u64;
                            write_sized(publication, p as *const libc::c_void, 8);
                        }
                    }
                }
                None => todo!(),
            }
        }
    }
}

#[cfg(feature = "testing")]
pub mod testing {

    pub use mosaiq_app_macros::{test, test_suite};

    use byteorder::ByteOrder;
    pub trait InspectIntercom {
        /// Return the amount of publications registered to the intercom.
        fn publications_len(&self) -> usize;

        /// Gets whether a publication for the given topic was created or not.
        fn get_publication(
            &self,
            topic: &'static str,
        ) -> Option<&mosaiq_app_c_bindings::MosaiqIntercomPublication>;
    }

    impl InspectIntercom for super::Intercom {
        fn publications_len(&self) -> usize {
            unsafe { (*self.publications.get()).len() }
        }

        fn get_publication(
            &self,
            topic: &'static str,
        ) -> Option<&mosaiq_app_c_bindings::MosaiqIntercomPublication> {
            unsafe {
                let publications = &*self.publications.get();
                match publications.get(topic) {
                    Some(p) => Some(&**p),
                    None => None,
                }
            }
        }
    }

    /// TestSystem is a mocked part of the Mosaiq cyclic system.
    /// It receives and holds publications and values sent via the intercom to allow users to
    /// verify their existence in tests.
    pub struct TestSystem {
        pub publications: Vec<mosaiq_app_c_bindings::MosaiqIntercomPublication>,
        pub published_values: Vec<Vec<u8>>,
    }

    impl TestSystem {
        /// Try to get a published u8 from the list of published values. Panics if it does not exist.
        pub fn get_value_u8(&self, index: usize) -> u8 {
            self.published_values[index][0]
        }

        /// Try to get a published u16 from the list of published values. Panics if it does not exist.
        pub fn get_value_u16(&self, index: usize) -> u16 {
            byteorder::LittleEndian::read_u16(&self.published_values[index])
        }

        /// Try to get a published u32 from the list of published values. Panics if it does not exist.
        pub fn get_value_u32(&self, index: usize) -> u32 {
            byteorder::LittleEndian::read_u32(&self.published_values[index])
        }

        /// Try to get a published u64 from the list of published values. Panics if it does not exist.
        pub fn get_value_u64(&self, index: usize) -> u64 {
            byteorder::LittleEndian::read_u64(&self.published_values[index])
        }

        /// Try to get a published value as a byte array from the list of published values. Panics if it does not exist.
        pub fn val(&self, index: usize) -> &[u8] {
            &self.published_values[index]
        }

        /// Returns the amount of published values.
        pub fn values_len(&self) -> usize {
            self.published_values.len()
        }
    }
}
