#![forbid(unsafe_code)]

//! # morseclock - Yet another not-so-intuitive clock
//! 
//! morseclock allows you to generate a clock visualization inspired by [this hackaday post](https://hackaday.com/2016/07/29/a-one-led-clock/)
//! 
//! # Usage example
//! 
//! ```
//! use morseclock::{Clock, Format, MorseExt};
//! 
//! // Hour must be in the range [0, 23)
//! let hour = 16.try_into().unwrap();
//! // Minute must be in the range [0, 59)
//! let minute = 47.try_into().unwrap();
//! 
//! let clock = Clock::new(hour, minute, Format::Hour12);
//! 
//! let time: String = clock
//!     .into_iter()
//!     .morse()
//!     .collect();
//! 
//! assert_eq!(time, "--.=----")
//! ```
//! 
//! # How to read the "morsecode"
//! 
//! The mapping generated by the [`MorseExt`] iterator extension trait can be interpreted as follows
//! 
//! The string `"--.=----"` represents the time 4:47.  
//! The character `'='` separates the hour-hand from the minute-hand.
//! 
//! On the hour-hand a single `'-'` stands for a increment of of 3 hours (a quarter rotation of the hand), starting at the 0 o'clock position.  
//! A single `'.'` stands for an increment of 1 hour.  
//! 
//! On the minute-hand, a single `'-'` stands for a increment of 15 minutes (a quarter rotation of the hand), starting at the 0 o'clock position.
//! A single `'.'` stands for an increment of 5 minutes.
//! 
//! 
//! ## Some time examples on the 12 hour clock
//! 
//! 00:00 is `-=-`  
//! 03:00 is `--=-`  
//! 06:00 is `---=-`  
//! 21:40 is 09:40 is `----=---..`
//! 
//! All minutes are rounded to 5 minutes, therefore the str representations of the following times are equal
//! 
//! 00:00 is 00:01 is .. is 00:04

use std::error;
use std::fmt;
use std::iter;
use std::marker::PhantomData;

/// A collection of errors which can happen
#[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
pub enum Error {
    /// The given value is invalid for a hand
    InvalidHandValue,
}

impl fmt::Display for Error {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::InvalidHandValue => f.write_str("Invalid hand value"),
        }
    }
}

impl error::Error for Error {}

/// A trait describing the properties of a clock hand
pub trait ClockHand {
    /// The granularity of the given hand
    const GRANULARITY: u8;
    /// The maximum value of the given hand
    const MAX: u8;

    /// Converts a hand value into a number of long and short symbols
    fn to_long_short(value: u8) -> (u8, u8);
}

/// Hour marker struct for [`Hand`]
#[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
pub struct Hour;

impl ClockHand for Hour {
    const GRANULARITY: u8 = 1;
    const MAX: u8 = 24;

    fn to_long_short(value: u8) -> (u8, u8) {
        (value / 3 + 1, value % 3)
    }
}

/// Minute marker struct for [`Hand`]
#[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
pub struct Minute;

impl ClockHand for Minute {
    const GRANULARITY: u8 = 5;
    const MAX: u8 = 60;

    fn to_long_short(value: u8) -> (u8, u8) {
        (value / 15 + 1, value % 15 / Self::GRANULARITY)
    }
}

/// A clock hand
/// 
/// The only way to construct a hand is via the [`TryFrom`]/[`TryInto`] implementations
/// for [`Hand<Hour>`] and [`Hand<Minute>`]
/// 
/// # Example
/// ```
/// # use morseclock::{Hand, Minute, Hour};
/// # 
/// let hour = Hand::<Hour>::try_from(23);
/// let minute = Hand::<Minute>::try_from(59);
/// ```
#[derive(Clone, Copy, Debug, Hash)]
pub struct Hand<T> {
    value: u8,
    _marker: PhantomData<T>,
}

impl<H: ClockHand> PartialEq for Hand<H> {
    fn eq(&self, other: &Self) -> bool {
        self.value / H::GRANULARITY == other.value / H::GRANULARITY
    }
}

impl<H: ClockHand> Eq for Hand<H> {}

impl<H: ClockHand> TryFrom<u32> for Hand<H> {
    type Error = Error;

    fn try_from(value: u32) -> Result<Self, Self::Error> {
        if value >= H::MAX as u32 {
            return Err(Error::InvalidHandValue);
        }

        Ok(Hand {
            value: value as u8,
            _marker: PhantomData,
        })
    }
}

impl<H: ClockHand> IntoIterator for Hand<H> {
    type Item = Symbol;
    type IntoIter = HandIter<H>;

    fn into_iter(self) -> Self::IntoIter {
        let (long, short) = H::to_long_short(self.value);

        HandIter {
            long,
            short,
            _marker: PhantomData,
        }
    }
}

/// An iterator over a [`Hand`] which produces a series of [`Symbol`]s
pub struct HandIter<T> {
    long: u8,
    short: u8,
    _marker: PhantomData<T>,
}

impl<T> Iterator for HandIter<T> {
    type Item = Symbol;

    fn next(&mut self) -> Option<Self::Item> {
        if self.long > 0 {
            self.long -= 1;
            Some(Symbol::Long)
        } else if self.short > 0 {
            self.short -= 1;
            Some(Symbol::Short)
        } else {
            None
        }
    }
}

/// The symbols used to describe a time
#[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
pub enum Symbol {
    /// The break symbol (used to distinguish between hour and minute)
    Break,
    /// The short symbol
    Short,
    /// Thoe long symbol
    Long,
}

/// The output format of the clock, either 12 or 24 hours
#[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
pub enum Format {
    Hour12,
    Hour24,
}

/// A Clock which can produce a morse-like sequence of dits and dashes
#[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
pub struct Clock {
    pub hour: Hand<Hour>,
    pub minute: Hand<Minute>,
    pub format: Format,
}

impl Clock {
    pub fn new(hour: Hand<Hour>, minute: Hand<Minute>, format: Format) -> Self {
        Self {
            hour,
            minute,
            format,
        }
    }
}

impl IntoIterator for Clock {
    type Item = Symbol;
    type IntoIter = ClockIter;

    fn into_iter(mut self) -> Self::IntoIter {
        if self.format == Format::Hour12 {
            self.hour.value %= 12;
        }

        ClockIter(
            self.hour
                .into_iter()
                .chain(iter::once(Symbol::Break))
                .chain(self.minute.into_iter()),
        )
    }
}

/// An iterator over a [`Clock`] which produces a series of [`Symbol`]s
pub struct ClockIter(
    iter::Chain<iter::Chain<HandIter<Hour>, iter::Once<Symbol>>, HandIter<Minute>>,
);

impl Iterator for ClockIter {
    type Item = Symbol;

    fn next(&mut self) -> Option<Self::Item> {
        self.0.next()
    }
}

/// An extension trait for iterators which yield [`Symbol`]s
pub trait MorseExt {
    type Output;
    fn morse(self) -> Self::Output;
}

/// An iterator adapter which produces a series of morsecode-like symbols
///
/// See [`MorseExt`]
pub struct Morse<I>(I);

impl<I> MorseExt for I
where
    I: Iterator<Item = Symbol>,
{
    type Output = Morse<I>;

    fn morse(self) -> Self::Output {
        Morse(self)
    }
}

impl<I> Iterator for Morse<I>
where
    I: Iterator<Item = Symbol>,
{
    type Item = char;

    fn next(&mut self) -> Option<Self::Item> {
        match self.0.next() {
            Some(Symbol::Break) => Some('='),
            Some(Symbol::Short) => Some('.'),
            Some(Symbol::Long) => Some('-'),
            None => None,
        }
    }
}

#[cfg(test)]
pub mod tests {
    use super::*;

    macro_rules! eq {
        (($hour:expr, $minute:expr), $result_h12:expr) => {
            let hour = $hour.try_into().unwrap();
            let minute = $minute.try_into().unwrap();

            let dashes: String = Clock::new(hour, minute, Format::Hour12)
                .into_iter()
                .morse()
                .collect();

            assert_eq!(dashes, $result_h12);

            if $hour >= 12 {
                let dashes: String = Clock::new(hour, minute, Format::Hour24)
                    .into_iter()
                    .morse()
                    .collect();

                assert_eq!(dashes, concat!("----", $result_h12));
            }
        };
    }

    #[test]
    fn samples() {
        eq!((00, 00), "-=-");
        eq!((23, 59), "----..=----..");
        eq!((12, 00), "-=-");
        eq!((12, 15), "-=--");
        eq!((12, 30), "-=---");
        eq!((12, 34), "-=---");
        eq!((12, 35), "-=---.");
        eq!((14, 45), "-..=----");
        eq!((17, 47), "--..=----");
        eq!((18, 32), "---=---");
    }

    #[test]
    fn hand_equality() {
        assert_eq!(Hand::<Minute>::try_from(0), Hand::<Minute>::try_from(4));
    }
}
