use std::error::Error;
use std::fs;
use std::path::Path;

use rargsxd::*;
use serde::Deserialize;

#[derive(Deserialize)]
struct Config {
    outpath: Option<String>,
    dirformat: Option<String>,
    fileformat: Option<String>,
}

fn real_main() -> Result<(), Box<dyn Error>> {
    let mut config_path = dirs::config_dir().ok_or("No config directory")?;
    config_path.push("morg");
    config_path.push("config.toml");

    let mut args = ArgParser::from_argv0();
    args.author("BubbyRoosh")
        .info("Tool for organizing music by their metadata.")
        .version("0.1.0")
        .copyright("Copyright (c) 2021 BubbyRoosh")
        .usage("{name} [path/to/music ...]")
        .args(vec!(
            Arg::str("config", &config_path.to_string_lossy())
                .short('C')
                .long("config")
                .help("Path to the config file"),
            Arg::str("outpath", "")
                .short('o')
                .long("outpath")
                .help("\'outpath\' config override"),
            Arg::str("dirformat", "")
                .short('d')
                .long("dirformat")
                .help("\'dirformat\' config override"),
            Arg::str("fileformat", "")
                .short('f')
                .long("fileformat")
                .help("\'fileformat\' config override"),
        )).parse();

    println!("Parsing config...");
    let mut config: Config = toml::from_str(&fs::read_to_string(args.get_str("config"))?)?;
    if config.outpath.is_none() { 
        let music_dir = match dirs::audio_dir() {
            Some(ad) => ad.to_string_lossy().to_string(),
            None => {
                let mut home = dirs::home_dir().ok_or("")?;
                home.push("Music");
                home.to_string_lossy().to_string()
            }
        };
        config.outpath = Some(music_dir);
    }
    if config.dirformat.is_none() {
        config.dirformat = Some("%artist% - %album%".to_string());
    }
    if config.fileformat.is_none() {
        config.fileformat = Some("%artist% %tracknumber% - %title%".to_string());
    }
    if !args.get_str("outpath").is_empty() { config.outpath = Some(args.get_str("outpath")); }
    if !args.get_str("dirformat").is_empty() { config.outpath = Some(args.get_str("dirformat")); }
    if !args.get_str("fileformat").is_empty() { config.outpath = Some(args.get_str("fileformat")); }

    let outpath = config.outpath.unwrap().replace("$HOME", &dirs::home_dir().ok_or("")?.to_string_lossy().to_string());
    let dirformat = config.dirformat.unwrap();
    let fileformat = config.fileformat.unwrap();

    println!("Using \"{}\" dir.", outpath);
    fs::create_dir_all(&outpath)?;

    println!("Using \"{}\" directory format.", dirformat);
    println!("Using \"{}\" file format.", fileformat);

    for entry in args.extra {
        let organize_file = |filename: &str| -> Result<(), Box<dyn Error>> {
            let mut dir = Path::new(&outpath.clone()).to_path_buf();
            let mut format = dirformat.clone();
            let mut to = fileformat.clone();
            let tags = allaudiotags::AudioFile::from(Path::new(filename))?;
            if format.contains("%artist%") {format = format.replace("%artist%", &tags.get("ARTIST")?)}
            if format.contains("%album%") {format = format.replace("%album%", &tags.get("ALBUM")?)}
            if format.contains("%tracknumber%") {format = format.replace("%tracknumber%", &tags.get("TRACKNUMBER")?)}
            if format.contains("%title%") {format = format.replace("%title%", &tags.get("TITLE")?)}
            if format.contains("%date%") {format = format.replace("%date%", &tags.get("DATE")?)}
            dir.push(format);
            if !dir.exists() {fs::create_dir_all(&dir)?};

            if to.contains("%artist%") {to = to.replace("%artist%", &tags.get("ARTIST")?)}
            if to.contains("%album%") {to = to.replace("%album%", &tags.get("ALBUM")?)}
            if to.contains("%tracknumber%") {to = to.replace("%tracknumber%", &tags.get("TRACKNUMBER")?)}
            if to.contains("%title%") {to = to.replace("%title%", &tags.get("TITLE")?)}
            if to.contains("%date%") {to = to.replace("%date%", &tags.get("DATE")?)}
            to.push('.');
            to.push_str(&Path::new(filename).extension().unwrap().to_string_lossy().to_string());
            dir.push(to);
            fs::copy(filename, &dir)?;
            println!("{} -> {}", filename, dir.to_string_lossy().to_string());
            Ok(())
        };

        let path = Path::new(&entry);
        if !path.exists() {
            eprintln!("{} does not exist.", entry);
            continue;
        }
        if path.is_dir() {
            for direntry in path.read_dir()? {
                let direntry = direntry?;
                if direntry.metadata()?.is_file() {
                    let entry = direntry.path().to_string_lossy().to_string();
                    if let Err(e) = organize_file(&entry) {
                        eprintln!("Error organizing {}: {}", entry, e);
                    }
                }
            }
        } else {
            if let Err(e) = organize_file(&entry.clone()) {
                eprintln!("Error organizing {}: {}", entry, e);
            }
        }
    }

    Ok(())
}

fn main() {
    if let Err(e) = real_main() {
        eprintln!("Error: {}", e);
    }
}
